#!/usr/bin/env python3
"""
Chicago Food Inspection Violation Code Lookup System
Based on FDA Food Code and Chicago Department of Public Health standards

This module provides comprehensive violation code explanations for
article generation and reader education.
"""

from typing import Dict, Optional, List, Tuple
import re

class ViolationCodeLookup:
    """
    Comprehensive violation code lookup system for Chicago food inspections.
    Based on FDA Food Code 2017/2022 and Chicago CDPH standards.
    """
    
    def __init__(self):
        self.violation_codes = self._build_violation_database()
        self.severity_mapping = self._build_severity_mapping()
        self.category_mapping = self._build_category_mapping()
    
    def _build_violation_database(self) -> Dict[str, Dict]:
        """
        Build comprehensive violation code database.
        Based on FDA Food Code and Chicago CDPH standards.
        """
        return {
            # TEMPERATURE CONTROL VIOLATIONS
            "18": {
                "title": "Proper cooking time and temperature",
                "description": "Food must be cooked to safe minimum internal temperatures to kill harmful bacteria",
                "details": "Bacteria are destroyed at high temperatures. Cook foods to these minimum internal temperatures: 165°F for poultry, 160°F for ground meat, 145°F for whole cuts of beef/pork.",
                "category": "temperature",
                "severity": "priority",
                "health_risk": "high",
                "common_causes": ["Inadequate cooking time", "Equipment malfunction", "Staff training issues"],
                "prevention": "Use calibrated thermometers, train staff on proper cooking procedures"
            },
            
            "35": {
                "title": "Approved thawing methods used",
                "description": "Frozen foods must be thawed safely to prevent bacterial growth",
                "details": "Safe thawing methods: refrigerator (41°F or below), cold running water, microwave, or as part of cooking process. Never thaw at room temperature.",
                "category": "temperature",
                "severity": "priority_foundation",
                "health_risk": "medium",
                "common_causes": ["Thawing at room temperature", "Inadequate planning"],
                "prevention": "Plan ahead for safe thawing, train staff on approved methods"
            },
            
            # EQUIPMENT & UTENSIL VIOLATIONS
            "44": {
                "title": "Utensils, equipment and linens; properly stored, dried and handled",
                "description": "Clean equipment and utensils must be stored in sanitary conditions",
                "details": "Clean equipment and utensils must be stored in clean, dry locations, protected from contamination. Handle by areas that don't contact food.",
                "category": "equipment",
                "severity": "core",
                "health_risk": "low",
                "common_causes": ["Poor storage practices", "Contaminated storage areas"],
                "prevention": "Implement proper storage procedures, designate clean storage areas"
            },
            
            "47": {
                "title": "Food and non-food contact surfaces cleanable, properly designed, constructed and used",
                "description": "Equipment must be designed for easy cleaning and sanitation",
                "details": "Food contact surfaces must be smooth, non-absorbent, easily cleanable. Equipment must be properly maintained and used as designed.",
                "category": "equipment",
                "severity": "priority_foundation",
                "health_risk": "medium",
                "common_causes": ["Damaged equipment", "Improper equipment design", "Poor maintenance"],
                "prevention": "Regular equipment maintenance, proper equipment selection"
            },
            
            "48": {
                "title": "Warewashing facilities: installed, maintained, used; test strips",
                "description": "Dishwashing equipment must function properly with correct sanitizer levels",
                "details": "Warewashing equipment must maintain proper temperatures and sanitizer concentrations. Test strips must be available and used regularly.",
                "category": "sanitization",
                "severity": "priority",
                "health_risk": "high",
                "common_causes": ["Equipment malfunction", "Incorrect sanitizer concentration", "Missing test strips"],
                "prevention": "Regular equipment maintenance, monitor sanitizer levels, stock test strips"
            },
            
            # SANITATION VIOLATIONS
            "10": {
                "title": "Adequate handwashing facilities supplied & accessible",
                "description": "Handwashing stations must be properly equipped and accessible",
                "details": "Handwashing sinks must have hot and cold running water, soap, paper towels or air dryer. Must be accessible to employees at all times.",
                "category": "sanitation",
                "severity": "priority",
                "health_risk": "high",
                "common_causes": ["Blocked access", "Missing supplies", "Equipment failure"],
                "prevention": "Regular supply checks, keep areas clear, maintain equipment"
            },
            
            # COMMON CHICAGO VIOLATION CODES
            "51": {
                "title": "Plumbing installed; proper backflow devices",
                "description": "Plumbing systems must prevent contamination and backflow",
                "details": "Plumbing must be installed to prevent cross-connections and backflow. Backflow prevention devices required where applicable.",
                "category": "plumbing",
                "severity": "priority_foundation",
                "health_risk": "medium",
                "common_causes": ["Improper installation", "Missing backflow devices", "Cross-connections"],
                "prevention": "Professional plumbing installation, regular inspections"
            },
            
            "52": {
                "title": "Refuse, recyclables properly removed",
                "description": "Garbage and recyclables must be properly stored and removed",
                "details": "Refuse must be stored in leak-proof containers with tight-fitting lids, removed frequently enough to prevent pest attraction and odors.",
                "category": "waste_management",
                "severity": "core",
                "health_risk": "low",
                "common_causes": ["Infrequent removal", "Improper containers", "Poor storage practices"],
                "prevention": "Regular waste removal schedule, proper containers, staff training"
            },
            
            "53": {
                "title": "Physical facilities installed, maintained & clean",
                "description": "Building and facilities must be maintained in good condition",
                "details": "Floors, walls, ceilings must be smooth, easily cleanable, and in good repair. Facilities must be kept clean and well-maintained.",
                "category": "facilities",
                "severity": "core",
                "health_risk": "low",
                "common_causes": ["Poor maintenance", "Damaged surfaces", "Inadequate cleaning"],
                "prevention": "Regular maintenance schedule, prompt repairs, thorough cleaning"
            }
        }
    
    def _build_severity_mapping(self) -> Dict[str, str]:
        """Map violation severity levels to human-readable descriptions."""
        return {
            "priority": "Critical - Immediate health hazard that must be corrected immediately",
            "priority_foundation": "Serious - Potential health hazard that must be corrected within specified timeframe", 
            "core": "Minor - Does not pose immediate threat to public health"
        }
    
    def _build_category_mapping(self) -> Dict[str, str]:
        """Map violation categories to descriptions."""
        return {
            "temperature": "Temperature Control",
            "equipment": "Equipment & Utensils", 
            "sanitation": "Sanitation & Hygiene",
            "plumbing": "Plumbing & Water Systems",
            "waste_management": "Waste Management",
            "facilities": "Physical Facilities",
            "food_handling": "Food Handling & Storage",
            "employee_health": "Employee Health & Hygiene"
        }
    
    def lookup_violation(self, code: str) -> Optional[Dict]:
        """
        Look up violation code and return comprehensive information.
        
        Args:
            code: Violation code (e.g., "52", "18", "44")
            
        Returns:
            Dictionary with violation information or None if not found
        """
        # Clean the code - remove any non-numeric characters for lookup
        clean_code = re.sub(r'[^\d]', '', str(code))
        
        if clean_code in self.violation_codes:
            violation = self.violation_codes[clean_code].copy()
            violation['code'] = clean_code
            violation['severity_description'] = self.severity_mapping.get(violation['severity'], 'Unknown')
            violation['category_description'] = self.category_mapping.get(violation['category'], 'Other')
            return violation
        
        return None
    
    def parse_violations_text(self, violations_text: str) -> List[Dict]:
        """
        Parse violations text from Chicago inspection data and return violation details.
        
        Args:
            violations_text: Raw violations text from inspection record
            
        Returns:
            List of violation dictionaries with explanations
        """
        if not violations_text or violations_text.strip() == '':
            return []
        
        violations = []
        
        # Extract violation codes using regex
        # Look for patterns like "52. REFUSE" or "18 - TEMPERATURE" etc.
        code_pattern = r'\b(\d{1,3})\b'
        codes = re.findall(code_pattern, violations_text)
        
        for code in codes:
            violation_info = self.lookup_violation(code)
            if violation_info:
                violations.append(violation_info)
        
        return violations
    
    def explain_violations_for_article(self, violations_text: str) -> str:
        """
        Generate reader-friendly explanation of violations for article content.
        
        Args:
            violations_text: Raw violations text from inspection record
            
        Returns:
            Formatted explanation suitable for article content
        """
        violations = self.parse_violations_text(violations_text)
        
        if not violations:
            return "No specific violations were detailed in the inspection report."
        
        explanations = []
        
        for violation in violations:
            severity_emoji = {
                'priority': '🚨',
                'priority_foundation': '⚠️',
                'core': '📋'
            }.get(violation['severity'], '📋')
            
            explanation = f"{severity_emoji} **{violation['title']}** (Code {violation['code']}): {violation['description']}"
            
            if violation['health_risk'] == 'high':
                explanation += " This is considered a high-risk violation that could pose immediate health dangers."
            elif violation['health_risk'] == 'medium':
                explanation += " This violation creates potential health risks if not addressed."
            
            explanations.append(explanation)
        
        return "\n\n".join(explanations)
    
    def get_violation_summary(self, violations_text: str) -> Dict[str, any]:
        """
        Get summary statistics about violations for article metadata.
        
        Args:
            violations_text: Raw violations text from inspection record
            
        Returns:
            Dictionary with violation summary statistics
        """
        violations = self.parse_violations_text(violations_text)
        
        if not violations:
            return {
                'total_violations': 0,
                'priority_count': 0,
                'priority_foundation_count': 0,
                'core_count': 0,
                'high_risk_count': 0,
                'categories': [],
                'worst_severity': 'none'
            }
        
        summary = {
            'total_violations': len(violations),
            'priority_count': len([v for v in violations if v['severity'] == 'priority']),
            'priority_foundation_count': len([v for v in violations if v['severity'] == 'priority_foundation']),
            'core_count': len([v for v in violations if v['severity'] == 'core']),
            'high_risk_count': len([v for v in violations if v['health_risk'] == 'high']),
            'categories': list(set([v['category'] for v in violations])),
            'worst_severity': 'core'
        }
        
        # Determine worst severity
        if summary['priority_count'] > 0:
            summary['worst_severity'] = 'priority'
        elif summary['priority_foundation_count'] > 0:
            summary['worst_severity'] = 'priority_foundation'
        
        return summary
    
    def get_violation_tags(self, violations_text: str) -> List[str]:
        """
        Generate tags for auto-tagging system based on violations.
        
        Args:
            violations_text: Raw violations text from inspection record
            
        Returns:
            List of tags for article auto-tagging
        """
        violations = self.parse_violations_text(violations_text)
        tags = []
        
        if not violations:
            return tags
        
        # Add category tags
        categories = set([v['category'] for v in violations])
        for category in categories:
            category_name = self.category_mapping.get(category, category)
            tags.append(f"violation-{category}")
            tags.append(category_name.lower().replace(' ', '-'))
        
        # Add severity tags
        if any(v['severity'] == 'priority' for v in violations):
            tags.extend(['critical-violations', 'immediate-closure-risk'])
        elif any(v['severity'] == 'priority_foundation' for v in violations):
            tags.extend(['serious-violations', 'health-risk'])
        
        # Add specific violation type tags
        for violation in violations:
            if violation['code'] in ['18', '35']:  # Temperature violations
                tags.append('temperature-violations')
            elif violation['code'] in ['44', '47', '48']:  # Equipment violations
                tags.append('equipment-violations')
            elif violation['code'] in ['10']:  # Sanitation violations
                tags.append('sanitation-violations')
        
        return list(set(tags))  # Remove duplicates

# Global instance for easy import
violation_lookup = ViolationCodeLookup()

# Convenience functions for easy use
def explain_violation(code: str) -> Optional[str]:
    """Quick violation explanation lookup."""
    violation = violation_lookup.lookup_violation(code)
    return violation['description'] if violation else None

def explain_violations_text(violations_text: str) -> str:
    """Quick violations text explanation."""
    return violation_lookup.explain_violations_for_article(violations_text)

def get_violation_tags(violations_text: str) -> List[str]:
    """Quick violation tags generation."""
    return violation_lookup.get_violation_tags(violations_text)

if __name__ == "__main__":
    # Test the system
    lookup = ViolationCodeLookup()
    
    # Test individual code lookup
    print("Testing violation code 52:")
    violation = lookup.lookup_violation("52")
    if violation:
        print(f"Title: {violation['title']}")
        print(f"Description: {violation['description']}")
        print(f"Severity: {violation['severity_description']}")
        print()
    
    # Test violations text parsing
    sample_text = "52. REFUSE, RECYCLABLES PROPERLY REMOVED | Comments: Observed dumpster lids open"
    print("Testing violations text parsing:")
    print(f"Input: {sample_text}")
    print(f"Explanation: {lookup.explain_violations_for_article(sample_text)}")
    print(f"Tags: {lookup.get_violation_tags(sample_text)}")