#!/usr/bin/env python3
"""
Test article display on website
"""

import requests
import re
from bs4 import BeautifulSoup

print("=" * 60)
print("TESTING ARTICLE DISPLAY")
print("=" * 60)

# Test 1: Check homepage
print("\n1. Testing Homepage...")
response = requests.get("http://160.153.178.131/_sites/cleankitchens/")
if response.status_code == 200:
    soup = BeautifulSoup(response.text, 'html.parser')
    
    # Check for article links
    article_links = soup.find_all('a', href=re.compile(r'/(2025|2024)/'))
    print(f"   Found {len(article_links)} date-based article links")
    
    # Check for article cards
    cards = soup.find_all(class_=re.compile('article-card'))
    print(f"   Found {len(cards)} article cards")
    
    # Look for our specific articles
    if '90-miles' in response.text.lower():
        print("   ✅ Found 90 Miles Cuban Cafe article")
    else:
        print("   ❌ 90 Miles Cuban Cafe article NOT found")
    
    if 'soule' in response.text.lower():
        print("   ✅ Found Soule Chicago article")
    else:
        print("   ❌ Soule Chicago article NOT found")
    
    # Check what's in the featured section
    featured = soup.find(class_='featured-article')
    if featured:
        title = featured.find('h2') or featured.find('h3')
        if title:
            print(f"   Featured article: {title.text.strip()[:50]}...")
    
    # Check recent sections
    sections = soup.find_all('section')
    for section in sections[:3]:
        h2 = section.find('h2')
        if h2:
            articles = section.find_all('article')
            print(f"   Section '{h2.text.strip()}': {len(articles)} articles")

# Test 2: Direct article access
print("\n2. Testing Direct Article Access...")
test_slugs = [
    "2025/08/16/soule-chicago-corp-chicago-restaurant-putting-diners-at-ris",
    "2025/08/16/90-miles-cuban-cafe-chicago-restaurant-with-sewage-leak-and-"
]

for slug in test_slugs:
    url = f"http://160.153.178.131/_sites/cleankitchens/{slug}"
    response = requests.get(url)
    
    print(f"\n   Testing: /{slug[:40]}...")
    print(f"   Status: {response.status_code}")
    
    if response.status_code == 200:
        soup = BeautifulSoup(response.text, 'html.parser')
        title = soup.find('title')
        if title:
            title_text = title.text.strip()
            print(f"   Title: {title_text[:60]}...")
            
            if 'not found' in title_text.lower():
                print("   ❌ Showing 404 page")
            elif 'portillo' in response.text.lower() and 'portillo' not in slug:
                print("   ⚠️  Showing fallback Portillo's article")
            else:
                print("   ✅ Showing correct article!")

# Test 3: Check PHP error log for issues
print("\n3. Checking for PHP errors...")
import subprocess
result = subprocess.run(['tail', '-n', '20', '/var/log/apache2/error.log'], 
                       capture_output=True, text=True)
php_errors = [line for line in result.stdout.split('\n') if 'cleankitchens' in line.lower()]
if php_errors:
    print("   Recent PHP errors:")
    for error in php_errors[-3:]:
        print(f"   {error[:100]}...")
else:
    print("   No recent PHP errors")

print("\n" + "=" * 60)
print("RECOMMENDATIONS:")
print("=" * 60)

print("""
If articles aren't displaying:
1. Check that Weaviate queries handle the full slug path
2. Verify PHP error_log for query issues
3. Test GraphQL queries directly
4. Check Redis cache might have old data
""")