#!/bin/bash

# CleanKitchens Background Processing Startup Script
# This will run the processing in the background and survive SSH disconnections

SCRIPT_DIR="/var/www/twin-digital-media/public_html/_sites/cleankitchens/production/scripts"
LOG_DIR="/var/www/twin-digital-media/public_html/_sites/cleankitchens/production/logs"
PYTHON_ENV="/home/chris/cleankitchens-env/bin/python3"

# Create log directory if it doesn't exist
mkdir -p "$LOG_DIR"

# Set up log files
PROCESSING_LOG="$LOG_DIR/processing.log"
STARTUP_LOG="$LOG_DIR/startup.log"

echo "🚀 Starting CleanKitchens processing at $(date)" | tee -a "$STARTUP_LOG"
echo "📁 Script directory: $SCRIPT_DIR" | tee -a "$STARTUP_LOG"
echo "📁 Log directory: $LOG_DIR" | tee -a "$STARTUP_LOG"
echo "🐍 Python environment: $PYTHON_ENV" | tee -a "$STARTUP_LOG"

# Check if already running
if pgrep -f "process_recent_years_auto.py" > /dev/null; then
    echo "⚠️  Processing already running. PID: $(pgrep -f process_recent_years_auto.py)" | tee -a "$STARTUP_LOG"
    echo "📋 To check progress: tail -f $PROCESSING_LOG"
    echo "🛑 To stop: pkill -f process_recent_years_auto.py"
    exit 1
fi

# Start the processing in background with nohup
echo "🔄 Starting background processing..." | tee -a "$STARTUP_LOG"

cd "$SCRIPT_DIR"

# Use nohup to ensure it survives SSH disconnection
nohup "$PYTHON_ENV" process_recent_years_auto.py > "$PROCESSING_LOG" 2>&1 &

PROCESS_PID=$!
echo "✅ Process started with PID: $PROCESS_PID" | tee -a "$STARTUP_LOG"
echo "$PROCESS_PID" > "$LOG_DIR/processing.pid"

echo ""
echo "🎉 CleanKitchens processing started successfully!"
echo ""
echo "📋 Monitor progress:"
echo "   tail -f $PROCESSING_LOG"
echo ""
echo "🔍 Check status:"
echo "   ps aux | grep process_recent_years_auto"
echo ""
echo "🛑 Stop processing:"
echo "   pkill -f process_recent_years_auto.py"
echo ""
echo "📊 The process will generate ~24,000 articles from 2023-2025 data"
echo "💰 Estimated cost: ~$24.52"
echo "⏱️  Estimated time: 3-4 hours"
echo ""
echo "📁 All logs saved to: $LOG_DIR"