#!/usr/bin/env python3
"""
Image Optimization Helper for CleanKitchens
Converts existing images to optimized formats:
1. Social media sizes (1200x630 for OpenGraph, 1080x1080 for Instagram)
2. WebP versions for page speed
3. Compressed JPEG versions
"""

import os
import sys
from PIL import Image, ImageOps
import glob

class ImageOptimizer:
    def __init__(self):
        self.base_path = '/var/www/twin-digital-media/public_html/_sites/cleankitchens/assets/images'
        
        # Social media dimensions
        self.social_sizes = {
            'og': (1200, 630),      # OpenGraph/Facebook
            'twitter': (1200, 675), # Twitter Card
            'instagram': (1080, 1080), # Instagram square
            'thumb': (400, 300)     # Thumbnail
        }
        
        # WebP quality settings
        self.webp_quality = 85
        self.jpeg_quality = 90
        
    def optimize_directory(self, directory_path):
        """Optimize all images in a directory"""
        print(f"Optimizing images in: {directory_path}")
        
        # Find all JPG files
        jpg_files = glob.glob(os.path.join(directory_path, "*.jpg"))
        
        for jpg_file in jpg_files:
            print(f"Processing: {os.path.basename(jpg_file)}")
            self.optimize_single_image(jpg_file)
    
    def optimize_single_image(self, image_path):
        """Optimize a single image file"""
        try:
            # Load the original image
            with Image.open(image_path) as img:
                # Convert to RGB if needed
                if img.mode != 'RGB':
                    img = img.convert('RGB')
                
                base_name = os.path.splitext(os.path.basename(image_path))[0]
                directory = os.path.dirname(image_path)
                
                # Create optimized subdirectory
                optimized_dir = os.path.join(directory, 'optimized')
                os.makedirs(optimized_dir, exist_ok=True)
                
                # Generate social media sizes
                for size_name, dimensions in self.social_sizes.items():
                    self.create_social_size(img, base_name, optimized_dir, size_name, dimensions)
                
                # Create WebP version of original
                webp_path = os.path.join(optimized_dir, f"{base_name}.webp")
                img.save(webp_path, 'WEBP', quality=self.webp_quality, optimize=True)
                print(f"  ✓ WebP: {os.path.basename(webp_path)}")
                
                # Create optimized JPEG
                jpg_path = os.path.join(optimized_dir, f"{base_name}_optimized.jpg")
                img.save(jpg_path, 'JPEG', quality=self.jpeg_quality, optimize=True)
                print(f"  ✓ JPEG: {os.path.basename(jpg_path)}")
                
        except Exception as e:
            print(f"  ✗ Error processing {image_path}: {e}")
    
    def create_social_size(self, img, base_name, output_dir, size_name, dimensions):
        """Create a specific social media size"""
        try:
            # Calculate the best crop
            img_copy = img.copy()
            
            # Use ImageOps.fit for smart cropping
            sized_img = ImageOps.fit(img_copy, dimensions, Image.Resampling.LANCZOS)
            
            # Save JPEG version
            jpg_path = os.path.join(output_dir, f"{base_name}_{size_name}.jpg")
            sized_img.save(jpg_path, 'JPEG', quality=self.jpeg_quality, optimize=True)
            
            # Save WebP version
            webp_path = os.path.join(output_dir, f"{base_name}_{size_name}.webp")
            sized_img.save(webp_path, 'WEBP', quality=self.webp_quality, optimize=True)
            
            print(f"  ✓ {size_name}: {dimensions[0]}x{dimensions[1]}")
            
        except Exception as e:
            print(f"  ✗ Error creating {size_name} size: {e}")
    
    def optimize_all_violations(self):
        """Optimize all violation images"""
        violations_dir = os.path.join(self.base_path, 'violations')
        if os.path.exists(violations_dir):
            self.optimize_directory(violations_dir)
        else:
            print(f"Directory not found: {violations_dir}")
    
    def optimize_all_chains(self):
        """Optimize all chain images"""
        chains_dir = os.path.join(self.base_path, 'chains')
        if os.path.exists(chains_dir):
            self.optimize_directory(chains_dir)
        else:
            print(f"Directory not found: {chains_dir}")
    
    def create_htaccess_webp_rules(self):
        """Create .htaccess rules for WebP serving"""
        htaccess_content = """
# WebP Image Serving
<IfModule mod_rewrite.c>
    RewriteEngine On
    
    # Check if browser supports WebP
    RewriteCond %{HTTP_ACCEPT} image/webp
    
    # Check if WebP version exists
    RewriteCond %{REQUEST_FILENAME} \.(jpe?g|png)$
    RewriteCond %{REQUEST_FILENAME}\.webp -f
    
    # Serve WebP version
    RewriteRule ^(.+)\.(jpe?g|png)$ $1.$2.webp [T=image/webp,E=accept:1]
</IfModule>

# Add WebP MIME type
<IfModule mod_mime.c>
    AddType image/webp .webp
</IfModule>

# Add Vary header for caching
<IfModule mod_headers.c>
    <FilesMatch "\.(jpe?g|png)$">
        Header append Vary Accept
    </FilesMatch>
</IfModule>
"""
        
        htaccess_path = os.path.join(self.base_path, '.htaccess')
        with open(htaccess_path, 'w') as f:
            f.write(htaccess_content.strip())
        
        print(f"✓ Created WebP serving rules: {htaccess_path}")
    
    def generate_usage_examples(self):
        """Generate usage examples for the optimized images"""
        print("\n=== Usage Examples ===")
        print("HTML with WebP fallback:")
        print("""
<picture>
    <source srcset="/assets/images/violations/optimized/rodent_1_og.webp" type="image/webp">
    <img src="/assets/images/violations/optimized/rodent_1_og.jpg" alt="Restaurant violation" 
         width="1200" height="630">
</picture>
""")
        
        print("OpenGraph meta tags:")
        print("""
<meta property="og:image" content="https://cleankitchens.com/assets/images/violations/optimized/rodent_1_og.jpg">
<meta property="og:image:width" content="1200">
<meta property="og:image:height" content="630">
""")

def main():
    optimizer = ImageOptimizer()
    
    print("=== CleanKitchens Image Optimizer ===")
    print("Creating optimized versions for social media and WebP...")
    
    # Optimize all images
    optimizer.optimize_all_violations()
    optimizer.optimize_all_chains()
    
    # Create WebP serving rules
    optimizer.create_htaccess_webp_rules()
    
    # Show usage examples
    optimizer.generate_usage_examples()
    
    print("\n=== Optimization Complete ===")
    print("Optimized images saved in 'optimized' subdirectories")
    print("Social media sizes: og (1200x630), twitter (1200x675), instagram (1080x1080), thumb (400x300)")
    print("Formats: JPEG (optimized) + WebP (compressed)")

if __name__ == "__main__":
    main()