#!/usr/bin/env python3
"""
Generate articles for all 10 inspections in the database
"""

import sys
import os
import time
sys.path.insert(0, '/var/www/twin-digital-media/public_html/_sites/cleankitchens/production/scripts')

from test_article_generator import TestArticleGenerator
import weaviate

def main():
    client = weaviate.connect_to_local(host="localhost", port=8080)
    
    try:
        # Get all inspections
        inspections_collection = client.collections.get("RawInspection")
        
        # Get the first 10 inspections
        all_inspections = inspections_collection.query.fetch_objects(
            return_properties=["inspection_id", "dba_name", "city", "results", "raw_data", "inspection_date"],
            limit=10
        )
        
        print(f"Found {len(all_inspections.objects)} inspections")
        
        # Check which already have articles
        articles_collection = client.collections.get("Articles")
        existing_articles = articles_collection.query.fetch_objects(
            return_properties=["inspection_id"],
            limit=100
        )
        
        existing_ids = set()
        for obj in existing_articles.objects:
            existing_ids.add(obj.properties.get('inspection_id'))
        
        print(f"Found {len(existing_ids)} existing articles")
        
        # Generate articles for all inspections
        generator = TestArticleGenerator()
        generated_count = 0
        skipped_count = 0
        
        for obj in all_inspections.objects:
            inspection = obj.properties
            inspection_id = inspection['inspection_id']
            
            if inspection_id in existing_ids:
                print(f"⏭️  Skipping {inspection['dba_name']} - article already exists")
                skipped_count += 1
                continue
            
            print(f"\n{'='*60}")
            print(f"Processing: {inspection['dba_name']}")
            print(f"Inspection ID: {inspection_id}")
            print(f"Result: {inspection['results']}")
            print(f"Date: {inspection['inspection_date']}")
            
            # Format for the generator
            formatted_inspection = {
                'inspection_id': inspection_id,
                'facility_name': inspection['dba_name'],
                'city': inspection['city'],
                'results': inspection['results'],
                'inspection_date': inspection['inspection_date'],
                'raw_data': inspection['raw_data'],
                'address': generator.extract_field_from_raw(inspection['raw_data'], 'Address'),
                'violations': generator.extract_violations_from_raw(inspection['raw_data']),
                'facility_type': generator.extract_field_from_raw(inspection['raw_data'], 'Facility Type'),
                'zip_code': generator.extract_field_from_raw(inspection['raw_data'], 'Zip')
            }
            
            # Generate article
            try:
                # Build Claude prompt
                prompt = generator.build_claude_prompt(formatted_inspection)
                
                # Call Claude API
                print("Calling Claude API...")
                claude_response = generator.call_claude_api(prompt)
                
                if not claude_response:
                    print("Failed to get Claude response")
                    continue
                
                # Extract sections
                meta_data, content = generator.extract_sections(claude_response)
                
                # Select image
                image_data = generator.select_image(formatted_inspection)
                
                # Save to Weaviate
                result = generator.save_article_to_weaviate(formatted_inspection, meta_data, content, image_data)
                
                if result:
                    generated_count += 1
                    print(f"✅ Article generated successfully")
                else:
                    print(f"❌ Failed to save article")
                
                # Small delay to avoid API rate limits
                time.sleep(2)
                
            except Exception as e:
                print(f"❌ Error generating article: {e}")
        
        print(f"\n{'='*60}")
        print(f"✅ Generation complete!")
        print(f"   Generated: {generated_count} new articles")
        print(f"   Skipped: {skipped_count} existing articles")
        print(f"   Total articles: {len(existing_ids) + generated_count}")
        
        # Clean up
        generator.weaviate_client.close()
        
    finally:
        client.close()

if __name__ == "__main__":
    main()