#!/usr/bin/env python3
"""
Check what's actually in the Articles collection
"""

import weaviate
import json

# Connect to Weaviate
client = weaviate.connect_to_local()
articles = client.collections.get("Articles")

# Get all articles
response = articles.query.fetch_objects(limit=10)

print("Articles in Weaviate:")
print("=" * 60)

for article in response.objects:
    props = article.properties
    print(f"\nTitle: {props.get('title', 'No title')}")
    print(f"Slug: {props.get('slug', 'No slug')}")
    print(f"City: {props.get('city', 'No city')}")
    print(f"State: {props.get('state', 'No state')}")
    print(f"Published: {props.get('published_date', 'No date')}")
    print(f"UUID: {article.uuid}")

print("\n" + "=" * 60)
print("Testing GraphQL query directly...")

# Test the exact query used by PHP
query = """
{
    Get {
        Articles(limit: 3, sort: [{path: ["published_date"], order: desc}]) {
            title
            slug
            excerpt
            city
            state
            published_date
            image_url
            establishment_name
        }
    }
}
"""

import requests
response = requests.post(
    'http://localhost:8080/v1/graphql',
    json={'query': query},
    headers={'Content-Type': 'application/json'}
)

if response.status_code == 200:
    data = response.json()
    if 'data' in data and 'Get' in data['data']:
        articles = data['data']['Get']['Articles']
        print(f"\nGraphQL returned {len(articles)} articles:")
        for article in articles:
            print(f"  - Slug: {article.get('slug')}")
else:
    print(f"GraphQL query failed: {response.status_code}")

client.close()