<?php
/**
 * Redis Cache Handler for CleanKitchens
 * Caches Weaviate queries to reduce latency and load
 */

class RedisCache {
    private $redis;
    private $connected = false;
    private $ttl = 3600; // 1 hour default TTL
    
    public function __construct() {
        try {
            $this->redis = new Redis();
            $this->connected = $this->redis->connect('127.0.0.1', 6379);
            // Use database 1 for cleankitchens (0 is default)
            if ($this->connected) {
                $this->redis->select(1);
            }
        } catch (Exception $e) {
            error_log("Redis connection failed: " . $e->getMessage());
            $this->connected = false;
        }
    }
    
    /**
     * Get cached data
     */
    public function get($key) {
        if (!$this->connected) return null;
        
        try {
            $data = $this->redis->get($this->makeKey($key));
            return $data ? json_decode($data, true) : null;
        } catch (Exception $e) {
            error_log("Redis get error: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Set cache data
     */
    public function set($key, $data, $ttl = null) {
        if (!$this->connected) return false;
        
        try {
            $ttl = $ttl ?? $this->ttl;
            return $this->redis->setex(
                $this->makeKey($key),
                $ttl,
                json_encode($data)
            );
        } catch (Exception $e) {
            error_log("Redis set error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Delete cached data
     */
    public function delete($key) {
        if (!$this->connected) return false;
        
        try {
            return $this->redis->del($this->makeKey($key));
        } catch (Exception $e) {
            error_log("Redis delete error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Clear all cache
     */
    public function flush() {
        if (!$this->connected) return false;
        
        try {
            return $this->redis->flushDB();
        } catch (Exception $e) {
            error_log("Redis flush error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Create namespaced key
     */
    private function makeKey($key) {
        return 'cleankitchens:' . $key;
    }
    
    /**
     * Check if Redis is available
     */
    public function isConnected() {
        return $this->connected;
    }
}

// Global cache instance
$cache = new RedisCache();