<?php
// Live functions that query Weaviate for real articles

// Include Redis cache
require_once __DIR__ . '/includes/cache.php';

/**
 * Get optimized image URL based on context
 * @param string $imageUrl Base image URL from database (e.g., '/assets/images/violations/closure_3.jpg')
 * @param string $context Context: 'page', 'og', 'twitter', 'thumb', 'instagram'
 * @return string Optimized image URL
 */
function getOptimizedImageUrl($imageUrl, $context = 'page', $article = null) {
    // Use single DALL-E generated professional inspection image for everything
    // This provides consistent branding across all articles
    $baseImagePath = '/assets/images/health-inspection-report';
    
    // Return appropriate format and size based on context
    switch($context) {
        case 'og':
        case 'twitter':
        case 'instagram':
        case 'schema':
            return $baseImagePath . '_og.jpg'; // Social sharing - 1200x630
        case 'thumb':
            return $baseImagePath . '_thumb.webp'; // Thumbnail - smaller size
        case 'page':
        default:
            return $baseImagePath . '_page.webp'; // Article page - optimized size
    }
}

/**
 * Generate image URL from category
 * @param string $category Image category from article (e.g., 'cleanliness', 'temperature')
 * @return string Image URL path
 */
function generateImageFromCategory($category) {
    if (empty($category)) {
        return '/assets/images/violations/general_1.jpg';
    }
    
    // Check for specific cuisines FIRST (highest priority)
    if (stripos($category, 'pizza') !== false || stripos($category, 'pizzeria') !== false) {
        // Use equipment/temperature images for pizza until we get specific ones
        return '/assets/images/violations/equipment_' . rand(1,4) . '.jpg';
    }
    if (stripos($category, 'chinese') !== false || stripos($category, 'asian') !== false || stripos($category, 'wok') !== false) {
        return '/assets/images/violations/temperature_' . rand(1,4) . '.jpg';
    }
    if (stripos($category, 'mexican') !== false || stripos($category, 'taco') !== false || stripos($category, 'burrito') !== false) {
        return '/assets/images/violations/cross_contamination_' . rand(1,4) . '.jpg';
    }
    if (stripos($category, 'italian') !== false || stripos($category, 'pasta') !== false) {
        return '/assets/images/violations/cleanliness_' . rand(1,4) . '.jpg';
    }
    if (stripos($category, 'burger') !== false || stripos($category, 'grill') !== false) {
        return '/assets/images/violations/equipment_' . rand(1,4) . '.jpg';
    }
    
    // Check for rodents (second priority)
    if (stripos($category, 'rodent') !== false || stripos($category, 'mice') !== false || stripos($category, 'rat') !== false) {
        $rodentImages = [
            '/assets/images/violations/rodent_1.jpg',
            '/assets/images/violations/rodent_2.jpg',
            '/assets/images/violations/rodent_3.jpg',
            '/assets/images/violations/rodent_4.jpg'
        ];
        return $rodentImages[array_rand($rodentImages)];
    }
    
    // Check for roaches (third priority)
    if (stripos($category, 'roach') !== false || stripos($category, 'cockroach') !== false) {
        $roachImages = [
            '/assets/images/violations/roach_1.jpg',
            '/assets/images/violations/roach_2.jpg',
            '/assets/images/violations/roach_3.jpg',
            '/assets/images/violations/roach_4.jpg'
        ];
        return $roachImages[array_rand($roachImages)];
    }
    
    // Check for specific violation types (medium priority)
    if (stripos($category, 'temperature') !== false) {
        return '/assets/images/violations/temperature_' . rand(1,4) . '.jpg';
    }
    if (stripos($category, 'cleanliness') !== false || stripos($category, 'clean') !== false) {
        return '/assets/images/violations/cleanliness_' . rand(1,4) . '.jpg';
    }
    if (stripos($category, 'equipment') !== false) {
        return '/assets/images/violations/equipment_' . rand(1,4) . '.jpg';
    }
    if (stripos($category, 'handwashing') !== false || stripos($category, 'handwash') !== false) {
        return '/assets/images/violations/handwashing_' . rand(1,4) . '.jpg';
    }
    if (stripos($category, 'plumbing') !== false) {
        return '/assets/images/violations/plumbing_' . rand(1,4) . '.jpg';
    }
    if (stripos($category, 'mold') !== false) {
        return '/assets/images/violations/mold_' . rand(1,4) . '.jpg';
    }
    if (stripos($category, 'structural') !== false || stripos($category, 'structure') !== false) {
        return '/assets/images/violations/structural_' . rand(1,4) . '.jpg';
    }
    if (stripos($category, 'lighting') !== false) {
        return '/assets/images/violations/lighting_' . rand(1,4) . '.jpg';
    }
    if (stripos($category, 'garbage') !== false || stripos($category, 'waste') !== false) {
        return '/assets/images/violations/garbage_' . rand(1,4) . '.jpg';
    }
    if (stripos($category, 'cross') !== false || stripos($category, 'contamination') !== false) {
        return '/assets/images/violations/cross_contamination_' . rand(1,4) . '.jpg';
    }
    
    // Default: random from all violation images
    $allImages = [
        '/assets/images/violations/temperature_1.jpg', '/assets/images/violations/temperature_2.jpg', '/assets/images/violations/temperature_3.jpg', '/assets/images/violations/temperature_4.jpg',
        '/assets/images/violations/cleanliness_1.jpg', '/assets/images/violations/cleanliness_2.jpg', '/assets/images/violations/cleanliness_3.jpg', '/assets/images/violations/cleanliness_4.jpg',
        '/assets/images/violations/equipment_1.jpg', '/assets/images/violations/equipment_2.jpg', '/assets/images/violations/equipment_3.jpg', '/assets/images/violations/equipment_4.jpg',
        '/assets/images/violations/handwashing_1.jpg', '/assets/images/violations/handwashing_2.jpg', '/assets/images/violations/handwashing_3.jpg', '/assets/images/violations/handwashing_4.jpg',
        '/assets/images/violations/plumbing_1.jpg', '/assets/images/violations/plumbing_2.jpg', '/assets/images/violations/plumbing_3.jpg', '/assets/images/violations/plumbing_4.jpg',
        '/assets/images/violations/structural_1.jpg', '/assets/images/violations/structural_2.jpg', '/assets/images/violations/structural_3.jpg', '/assets/images/violations/structural_4.jpg',
        '/assets/images/violations/lighting_1.jpg', '/assets/images/violations/lighting_2.jpg', '/assets/images/violations/lighting_3.jpg', '/assets/images/violations/lighting_4.jpg',
        '/assets/images/violations/mold_1.jpg', '/assets/images/violations/mold_2.jpg', '/assets/images/violations/mold_3.jpg', '/assets/images/violations/mold_4.jpg',
        '/assets/images/violations/garbage_1.jpg', '/assets/images/violations/garbage_2.jpg', '/assets/images/violations/garbage_3.jpg', '/assets/images/violations/garbage_4.jpg',
        '/assets/images/violations/cross_contamination_1.jpg', '/assets/images/violations/cross_contamination_2.jpg', '/assets/images/violations/cross_contamination_3.jpg', '/assets/images/violations/cross_contamination_4.jpg',
        '/assets/images/violations/general_1.jpg', '/assets/images/violations/general_2.jpg', '/assets/images/violations/general_3.jpg', '/assets/images/violations/general_4.jpg'
    ];
    
    return $allImages[array_rand($allImages)];
}

function connectToWeaviate() {
    // Simple curl-based connection to Weaviate
    $base_url = 'http://localhost:8080/v1';
    return $base_url;
}

function queryWeaviate($query) {
    $base_url = connectToWeaviate();
    $url = $base_url . '/graphql';
    
    $data = json_encode(['query' => $query]);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code === 200 && $response) {
        return json_decode($response, true);
    }
    
    return null;
}

function getFeaturedArticleFromDB() {
    // Query Articles collection for most recent article
    $query = '{
        Get {
            Articles(limit: 1, sort: [{path: ["published_date"], order: desc}]) {
                title
                slug
                excerpt
                city
                state
                published_date
                image_url
                image_alt
                establishment_name
            }
        }
    }';
    
    $response = queryWeaviate($query);
    
    if ($response && isset($response['data']['Get']['Articles'][0])) {
        $article = $response['data']['Get']['Articles'][0];
        return [
            'title' => $article['title'],
            'excerpt' => $article['excerpt'],
            'city' => $article['city'],
            'state' => $article['state'],
            'published_date' => $article['published_date'],
            'slug' => $article['slug'],
            'image_url' => $article['image_url'],
            'image_alt' => $article['image_alt'] ?? $article['title'],
            'formatted_time' => formatTimeAgo($article['published_date'])
        ];
    }
    
    // Fallback if no articles
    return [
        'title' => 'No Articles Available',
        'excerpt' => 'Check back soon for the latest restaurant health inspection news.',
        'city' => 'Chicago',
        'state' => 'IL',
        'published_date' => date('c'),
        'slug' => '',
        'image_url' => '/assets/images/violations/general_1.jpg',
        'image_alt' => 'Restaurant health inspection',
        'formatted_time' => 'Just now'
    ];
}


function getRecentClosuresFromDB() {
    // Query Articles for recent articles
    $query = '{
        Get {
            Articles(limit: 3, sort: [{path: ["published_date"], order: desc}]) {
                title
                slug
                excerpt
                city
                state
                published_date
                image_url
                establishment_name
            }
        }
    }';
    
    $response = queryWeaviate($query);
    
    if ($response && isset($response['data']['Get']['Articles'])) {
        $articles = [];
        foreach ($response['data']['Get']['Articles'] as $article) {
            // Calculate time ago for meta_label
            $time_ago = formatTimeAgo($article['published_date']);
            
            $articles[] = [
                'title' => $article['title'],
                'excerpt' => $article['excerpt'],
                'city' => $article['city'],
                'state' => $article['state'],
                'published_date' => $article['published_date'],
                'slug' => $article['slug'],
                'image_url' => $article['image_url'],
                'meta_label' => $time_ago // Add dynamic time ago as meta_label
            ];
        }
        return $articles;
    }
    
    // Fallback if no articles
    return [];
}

function getWeeklyArticlesFromDB() {
    // Since we have limited articles, just return the same articles with different labels
    // In production with more articles, this would use offset or filter by article_type
    $articles = getRecentClosuresFromDB();
    
    // Change the meta_label to indicate these are analysis/reports
    foreach ($articles as &$article) {
        // Vary the labels for different article types
        $labels = ['Analysis', 'Investigation', 'Weekly Report'];
        $article['meta_label'] = $labels[array_rand($labels)];
    }
    
    return $articles;
}

function getArticleBySlugFromDB($slug) {
    global $cache;
    
    try {
        // Clean the slug
        $cleanSlug = trim($slug);
        
        // Try to get from cache first (DISABLED FOR DEBUGGING)
        // $cacheKey = 'article:' . $cleanSlug;
        // $cached = $cache->get($cacheKey);
        // if ($cached !== null) {
        //     return $cached;
        // }
        
        // Query specific article by slug with all required fields
        $query = '{
            Get {
                Articles(where: {path: ["slug"], operator: Equal, valueText: "' . addslashes($cleanSlug) . '"}) {
                    title
                    slug
                    content
                    excerpt
                    meta_description
                    city
                    state
                    establishment_name
                    published_date
                    image_url
                    image_alt
                    auto_tags
                    inspection_id
                }
            }
        }';
        
        $response = queryWeaviate($query);
        
        // Check if we got a valid response with data
        if (!$response || !isset($response['data']['Get']['Articles']) || empty($response['data']['Get']['Articles'])) {
            return null;
        }
        
        $article = $response['data']['Get']['Articles'][0];
        
        // Validate required fields exist
        if (empty($article['title']) || empty($article['content']) || empty($article['slug'])) {
            return null;
        }
        
        // Extract address from article content since we don't have raw_data
        $addressData = [
            'latitude' => 41.8781,
            'longitude' => -87.6298,
            'city' => $article['city'] ?? 'Chicago',
            'state' => $article['state'] ?? 'IL',
            'streetAddress' => 'Restaurant Location',
            'postalCode' => '60601'
        ];
        
        // Generate comprehensive JSON-LD schema array (4 schemas)
        $schemas = [];
        
        // 1. Organization Schema
        $schemas[] = [
            "@context" => "https://schema.org",
            "@type" => "Organization",
            "name" => "CleanKitchens",
            "url" => "https://cleankitchens.org",
            "logo" => [
                "@type" => "ImageObject",
                "url" => "https://cleankitchens.org/logo.png",
                "width" => 600,
                "height" => 60
            ],
            "description" => "Real-time restaurant health inspection news and food safety reporting",
            "foundingDate" => "2025",
            "contactPoint" => [
                "@type" => "ContactPoint",
                "contactType" => "editorial",
                "email" => "editorial@cleankitchens.org"
            ],
            "sameAs" => [
                "https://twitter.com/cleankitchens",
                "https://facebook.com/cleankitchens"
            ]
        ];
        
        // 2. NewsArticle Schema
        $schemas[] = [
            "@context" => "https://schema.org",
            "@type" => "NewsArticle",
            "headline" => $article['title'],
            "description" => $article['meta_description'] ?? substr(strip_tags($article['content']), 0, 160),
            "articleBody" => strip_tags($article['content']),
            "datePublished" => $article['published_date'],
            "dateModified" => $article['published_date'],
            "author" => [
                "@type" => "Organization",
                "name" => "CleanKitchens Editorial Team",
                "url" => "https://cleankitchens.org"
            ],
            "publisher" => [
                "@type" => "Organization",
                "name" => "CleanKitchens",
                "url" => "https://cleankitchens.org",
                "logo" => [
                    "@type" => "ImageObject",
                    "url" => "https://cleankitchens.org/logo.png"
                ]
            ],
            "mainEntityOfPage" => [
                "@type" => "WebPage",
                "@id" => "https://cleankitchens.org/" . $article['slug']
            ],
            "image" => [
                "@type" => "ImageObject",
                "url" => "https://cleankitchens.org" . ($article['image_url'] ?? '/assets/images/violations/general_1.jpg'),
                "width" => 1200,
                "height" => 630
            ],
            "keywords" => implode(", ", is_array($article['auto_tags']) ? $article['auto_tags'] : ['Health Inspection'])
        ];
        
        // 3. Restaurant/LocalBusiness Schema
        $schemas[] = [
            "@context" => "https://schema.org",
            "@type" => "Restaurant",
            "name" => $article['establishment_name'] ?? 'Restaurant',
            "address" => [
                "@type" => "PostalAddress",
                "addressLocality" => $article['city'] ?? 'Chicago',
                "addressRegion" => $article['state'] ?? 'IL',
                "addressCountry" => "US"
            ],
            "geo" => [
                "@type" => "GeoCoordinates",
                "latitude" => $addressData['latitude'],
                "longitude" => $addressData['longitude']
            ],
            "servesCuisine" => "American",
            "priceRange" => "$$",
            "url" => "https://cleankitchens.org/" . $article['slug']
        ];
        
        // 4. FAQ Schema (common food safety questions)
        $schemas[] = [
            "@context" => "https://schema.org",
            "@type" => "FAQPage",
            "mainEntity" => [
                [
                    "@type" => "Question",
                    "name" => "What does a failed health inspection mean?",
                    "acceptedAnswer" => [
                        "@type" => "Answer",
                        "text" => "A failed health inspection means the restaurant had critical violations that pose immediate health risks. The establishment must correct violations before reopening."
                    ]
                ],
                [
                    "@type" => "Question", 
                    "name" => "How often are restaurants inspected?",
                    "acceptedAnswer" => [
                        "@type" => "Answer",
                        "text" => "High-risk restaurants are typically inspected 2-3 times per year, while lower-risk establishments may be inspected annually. Chicago inspects approximately 16,000 food establishments."
                    ]
                ],
                [
                    "@type" => "Question",
                    "name" => "Are violation codes standardized?",
                    "acceptedAnswer" => [
                        "@type" => "Answer",
                        "text" => "Yes, Chicago uses violation codes based on the FDA Food Code. Common codes include temperature violations (18), equipment issues (47-48), and cleanliness problems (various codes)."
                    ]
                ]
            ]
        ];
        
        // 5. Breadcrumb Schema
        $schemas[] = [
            "@context" => "https://schema.org",
            "@type" => "BreadcrumbList",
            "itemListElement" => [
                [
                    "@type" => "ListItem",
                    "position" => 1,
                    "name" => "Home",
                    "item" => "https://cleankitchens.org"
                ],
                [
                    "@type" => "ListItem",
                    "position" => 2,
                    "name" => $article['city'] ?? 'Chicago',
                    "item" => "https://cleankitchens.org/city/" . strtolower($article['city'] ?? 'chicago')
                ],
                [
                    "@type" => "ListItem",
                    "position" => 3,
                    "name" => "Health Inspections",
                    "item" => "https://cleankitchens.org/inspections"
                ],
                [
                    "@type" => "ListItem",
                    "position" => 4,
                    "name" => substr($article['title'], 0, 50) . "...",
                    "item" => "https://cleankitchens.org/" . $article['slug']
                ]
            ]
        ];

        // Build complete article array with schema
        $result = [
            'id' => $article['inspection_id'] ?? 'unknown',
            'title' => $article['title'],
            'slug' => $article['slug'],
            'content' => $article['content'],
            'excerpt' => $article['excerpt'] ?? '',
            'meta_title' => $article['title'],
            'meta_description' => $article['meta_description'] ?? '',
            'og_title' => $article['title'],
            'og_description' => $article['meta_description'] ?? '',
            'city' => $article['city'] ?? 'Chicago',
            'state' => $article['state'] ?? 'IL',
            'establishment_name' => $article['establishment_name'] ?? '',
            'published_date' => $article['published_date'] ?? date('c'),
            'image_url' => $article['image_url'] ?? '/assets/images/violations/general_1.jpg',
            'image_alt' => $article['image_alt'] ?? 'Restaurant health inspection violation',
            'tags' => is_array($article['auto_tags']) ? $article['auto_tags'] : ['Health Inspection'],
            'latitude' => $addressData['latitude'],
            'longitude' => $addressData['longitude'],
            'schema_json' => json_encode($schemas, JSON_UNESCAPED_SLASHES)
        ];
        
        // Cache the result for 1 hour (DISABLED FOR DEBUGGING)
        // $cache->set($cacheKey, $result, 3600);
        
        return $result;
        
    } catch (Exception $e) {
        error_log("Error in getArticleBySlugFromDB: " . $e->getMessage());
        return null;
    }
}

function getRelatedArticlesFromDB() {
    // Get other articles for related section
    $query = '{
        Get {
            Articles(limit: 3) {
                title
                slug
                city
                state
                published_date
            }
        }
    }';
    
    $response = queryWeaviate($query);
    
    if ($response && isset($response['data']['Get']['Articles'])) {
        $articles = [];
        foreach ($response['data']['Get']['Articles'] as $article) {
            $articles[] = [
                'title' => $article['title'],
                'slug' => $article['slug'],
                'city' => $article['city'],
                'state' => $article['state'],
                'published_date' => $article['published_date']
            ];
        }
        return $articles;
    }
    
    // Fallback
    return [];
}

// New improved time formatting function
function formatTimeAgo($timestamp) {
    $now = new DateTime();
    $then = new DateTime($timestamp);
    $diff = $now->diff($then);
    
    if ($diff->days == 0) {
        if ($diff->h == 0) {
            return "Just now";
        } elseif ($diff->h == 1) {
            return "1 hour ago";
        } else {
            return $diff->h . " hours ago";
        }
    } elseif ($diff->days == 1) {
        return "Yesterday";
    } elseif ($diff->days < 7) {
        return $diff->days . " days ago";
    } else {
        return $then->format('M j, Y');
    }
}

// Get dynamic homepage sections with rotating content
function getHomepageSections() {
    global $cache;
    
    // Try cache first (cache for 30 minutes since homepage updates frequently)
    $cacheKey = 'homepage:sections';
    $cached = $cache->get($cacheKey);
    if ($cached !== null) {
        return $cached;
    }
    
    $sections = [];
    
    // Recent Inspection Reports
    $recentQuery = '{
        Get {
            Articles(
                limit: 30
                sort: [{path: ["published_date"], order: desc}]
            ) {
                title slug excerpt city state published_date image_url image_category establishment_name
            }
        }
    }';
    
    $response = queryWeaviate($recentQuery);
    if ($response && isset($response['data']['Get']['Articles'])) {
        $allArticles = $response['data']['Get']['Articles'];
        
        // Track used article IDs to prevent duplicates
        $usedSlugs = [];
        
        // Helper function to get unique articles
        $getUniqueArticles = function($articles, $count) use (&$usedSlugs) {
            $unique = [];
            foreach ($articles as $article) {
                if (!in_array($article['slug'], $usedSlugs)) {
                    $unique[] = $article;
                    $usedSlugs[] = $article['slug'];
                    if (count($unique) >= $count) {
                        break;
                    }
                }
            }
            return $unique;
        };
        
        // Recent Inspections - get the most recent (no shuffle)
        $sections['recent'] = [
            'title' => 'Recent Inspection Reports',
            'articles' => $getUniqueArticles($allArticles, 6)
        ];
        
        // For remaining sections, shuffle remaining articles for variety
        $remainingArticles = array_filter($allArticles, function($article) use ($usedSlugs) {
            return !in_array($article['slug'], $usedSlugs);
        });
        shuffle($remainingArticles);
        
        // Food Safety Updates
        $sections['safety'] = [
            'title' => 'Food Safety Updates',
            'articles' => $getUniqueArticles($remainingArticles, 6)
        ];
        
        // Health Department Activity
        $sections['health'] = [
            'title' => 'Health Department Activity',
            'articles' => $getUniqueArticles($remainingArticles, 6)
        ];
        
        // This Week in Food Safety
        $sections['weekly'] = [
            'title' => 'This Week in Food Safety',
            'articles' => $getUniqueArticles($remainingArticles, 6)
        ];
        
        // Educational Resources
        $sections['education'] = [
            'title' => 'Educational Resources',
            'articles' => $getUniqueArticles($remainingArticles, 6)
        ];
    }
    
    // Add formatted time and optimized images to all articles
    foreach ($sections as &$section) {
        foreach ($section['articles'] as &$article) {
            $article['formatted_time'] = formatTimeAgo($article['published_date']);
            
            // Generate image URL from category if image_url is missing
            if (empty($article['image_url'])) {
                // Check restaurant name, title, and category for specific matches
                $searchText = ($article['establishment_name'] ?? '') . ' ' . ($article['title'] ?? '') . ' ' . ($article['image_category'] ?? '');
                $article['image_url'] = generateImageFromCategory($searchText);
            }
            
            $article['optimized_image'] = getOptimizedImageUrl($article['image_url'], 'thumb');
        }
    }
    
    // Cache for 30 minutes
    $cache->set($cacheKey, $sections, 1800);
    
    return $sections;
}

// Get tag data by slug
function getTagBySlug($slug) {
    $query = '{
        Get {
            TagPages(where: {path: ["slug"], operator: Equal, valueText: "' . $slug . '"}) {
                tag_name
                slug
                title
                description
                article_count
            }
        }
    }';
    
    $response = queryWeaviate($query);
    
    if ($response && isset($response['data']['Get']['TagPages'][0])) {
        return $response['data']['Get']['TagPages'][0];
    }
    
    return null;
}

// Get articles by tag name
function getArticlesByTag($tagName, $limit = 50) {
    $query = '{
        Get {
            Articles(
                where: {path: ["auto_tags"], operator: Equal, valueText: "' . $tagName . '"}
                limit: ' . $limit . '
                sort: [{path: ["published_date"], order: desc}]
            ) {
                title
                slug
                excerpt
                city
                state
                published_date
                image_url
                image_category
                establishment_name
                auto_tags
            }
        }
    }';
    
    $response = queryWeaviate($query);
    
    // Debug logging for tag articles
    error_log("getArticlesByTag query for '$tagName': " . json_encode($response));
    
    if ($response && isset($response['data']['Get']['Articles'])) {
        $articles = [];
        foreach ($response['data']['Get']['Articles'] as $article) {
            // Add formatted time and optimized images
            $article['formatted_time'] = formatTimeAgo($article['published_date']);
            
            // Generate image URL from category if image_url is missing
            if (empty($article['image_url'])) {
                $searchText = ($article['establishment_name'] ?? '') . ' ' . ($article['title'] ?? '') . ' ' . ($article['image_category'] ?? '');
                $article['image_url'] = generateImageFromCategory($searchText);
            }
            
            $article['optimized_image'] = getOptimizedImageUrl($article['image_url'], 'thumb');
            $articles[] = $article;
        }
        return $articles;
    }
    
    return [];
}

// Get paginated news archive
function getNewsArchive($page = 1) {
    $perPage = ($page <= 10) ? 20 : 50; // Rich format for first 10 pages, then compact
    $offset = ($page - 1) * $perPage;
    
    $query = '{
        Get {
            Articles(
                limit: ' . $perPage . '
                offset: ' . $offset . '
                sort: [{path: ["published_date"], order: desc}]
            ) {
                title
                slug
                excerpt
                city
                state
                published_date
                image_url
                establishment_name
            }
        }
    }';
    
    $response = queryWeaviate($query);
    
    if ($response && isset($response['data']['Get']['Articles'])) {
        $articles = $response['data']['Get']['Articles'];
        
        // Add formatted time to each article
        foreach ($articles as &$article) {
            $article['formatted_time'] = formatTimeAgo($article['published_date']);
            $article['formatted_date'] = date('M j, Y', strtotime($article['published_date']));
        }
        
        return [
            'articles' => $articles,
            'format' => ($page <= 10) ? 'cards' : 'list',
            'page' => $page,
            'per_page' => $perPage,
            'has_next' => count($articles) == $perPage
        ];
    }
    
    return [
        'articles' => [],
        'format' => 'cards',
        'page' => $page,
        'per_page' => $perPage,
        'has_next' => false
    ];
}

// Get total article count for pagination
function getTotalArticleCount() {
    $query = '{
        Aggregate {
            Articles {
                meta {
                    count
                }
            }
        }
    }';
    
    $response = queryWeaviate($query);
    
    if ($response && isset($response['data']['Aggregate']['Articles'][0]['meta']['count'])) {
        return $response['data']['Aggregate']['Articles'][0]['meta']['count'];
    }
    
    return 0;
}

function generateCompleteArticleSchema($article) {
    $baseUrl = 'http://160.153.178.131/_sites/cleankitchens';
    
    // Extract restaurant info from article data
    $restaurantName = $article['establishment_name'] ?? 'Restaurant';
    $address = extractAddressFromRawData($article);
    
    // Create comprehensive schema array
    $schemas = [
        // Main NewsArticle Schema
        [
            '@context' => 'https://schema.org',
            '@type' => ['NewsArticle', 'LearningResource'],
            '@id' => $baseUrl . '/' . $article['slug'],
            'headline' => $article['title'],
            'description' => $article['meta_description'],
            'image' => [
                '@type' => 'ImageObject',
                'url' => $baseUrl . $article['image_url'],
                'width' => 1200,
                'height' => 630
            ],
            'datePublished' => $article['published_date'],
            'dateModified' => $article['published_date'],
            'author' => [
                '@type' => 'Organization',
                'name' => 'CleanKitchens Editorial Team'
            ],
            'publisher' => [
                '@type' => 'Organization',
                'name' => 'CleanKitchens',
                'logo' => [
                    '@type' => 'ImageObject',
                    'url' => $baseUrl . '/logo.png'
                ],
                'foundingDate' => '2024',
                'sameAs' => [
                    'https://twitter.com/cleankitchens',
                    'https://facebook.com/cleankitchens'
                ]
            ],
            'mainEntityOfPage' => $baseUrl . '/' . $article['slug'],
            'about' => [
                '@type' => 'Restaurant',
                '@id' => '#restaurant-' . md5($restaurantName . $address['streetAddress']),
                'name' => $restaurantName,
                'address' => $address
            ],
            'mentions' => [
                [
                    '@type' => 'Event',
                    'name' => 'Health Inspection',
                    'startDate' => date('Y-m-d', strtotime($article['published_date'])),
                    'description' => 'Restaurant health inspection conducted by Chicago Department of Public Health',
                    'location' => [
                        '@type' => 'Place',
                        'name' => $restaurantName,
                        'address' => $address
                    ]
                ]
            ],
            'educationalUse' => 'Food Safety Education',
            'teaches' => 'Restaurant Health Standards',
            'learningResourceType' => 'Case Study',
            'keywords' => 'food safety, health violation, restaurant inspection, ' . $article['city'],
            'articleSection' => 'Health Violations',
            'wordCount' => str_word_count(strip_tags($article['content']))
        ],
        
        // BreadcrumbList Schema
        [
            '@context' => 'https://schema.org',
            '@type' => 'BreadcrumbList',
            'itemListElement' => [
                [
                    '@type' => 'ListItem',
                    'position' => 1,
                    'name' => 'Home',
                    'item' => $baseUrl . '/'
                ],
                [
                    '@type' => 'ListItem',
                    'position' => 2,
                    'name' => $article['city'],
                    'item' => $baseUrl . '/city/' . strtolower($article['city'])
                ],
                [
                    '@type' => 'ListItem',
                    'position' => 3,
                    'name' => $article['title'],
                    'item' => $baseUrl . '/' . $article['slug']
                ]
            ]
        ],
        
        // Restaurant Schema
        [
            '@context' => 'https://schema.org',
            '@type' => 'Restaurant',
            '@id' => '#restaurant-' . md5($restaurantName . $address['streetAddress']),
            'name' => $restaurantName,
            'address' => $address,
            'servesCuisine' => 'American', // Could be extracted from article content
            'priceRange' => '$' // Default assumption
        ],
        
        // FAQ Schema - Standard questions for all articles
        [
            '@context' => 'https://schema.org',
            '@type' => 'FAQPage',
            'mainEntity' => [
                [
                    '@type' => 'Question',
                    'name' => 'When was this restaurant last inspected?',
                    'acceptedAnswer' => [
                        '@type' => 'Answer',
                        'text' => 'This restaurant was inspected on ' . date('F j, Y', strtotime($article['published_date'])) . ' by the ' . $article['city'] . ' health department.'
                    ]
                ],
                [
                    '@type' => 'Question',
                    'name' => 'What was the inspection result?',
                    'acceptedAnswer' => [
                        '@type' => 'Answer',
                        'text' => 'The restaurant failed its health inspection due to critical violations that posed immediate health risks to customers.'
                    ]
                ],
                [
                    '@type' => 'Question',
                    'name' => 'What does a failed inspection mean?',
                    'acceptedAnswer' => [
                        '@type' => 'Answer',
                        'text' => 'A failed inspection in Chicago\'s Pass/Fail system indicates critical food safety violations that pose immediate health risks and require immediate correction.'
                    ]
                ],
                [
                    '@type' => 'Question',
                    'name' => 'What violations were found?',
                    'acceptedAnswer' => [
                        '@type' => 'Answer',
                        'text' => 'The specific violations included issues with food temperature control, cleanliness standards, and proper food handling procedures. Details are provided in the full inspection report.'
                    ]
                ]
            ]
        ]
    ];
    
    return json_encode($schemas, JSON_UNESCAPED_SLASHES);
}

function extractAddressFromRawData($article) {
    // Extract address components from raw data or use defaults
    $defaultAddress = [
        '@type' => 'PostalAddress',
        'streetAddress' => 'Chicago Location',
        'addressLocality' => $article['city'] ?? 'Chicago',
        'addressRegion' => $article['state'] ?? 'IL',
        'postalCode' => '60601', // Default Chicago zip
        'addressCountry' => 'US'
    ];
    
    // If we have inspection data, try to parse the address
    if (isset($article['raw_data']) && !empty($article['raw_data'])) {
        $rawData = $article['raw_data'];
        
        // Try to extract address from raw data
        if (preg_match('/Address:([^,]+)/', $rawData, $matches)) {
            $defaultAddress['streetAddress'] = trim($matches[1]);
        }
        
        // Try to extract zip code
        if (preg_match('/Zip:(\d{5})/', $rawData, $matches)) {
            $defaultAddress['postalCode'] = $matches[1];
        }
    }
    
    return $defaultAddress;
}

function extractAddressFromContent($article) {
    // Extract location data from article content and metadata
    $city = $article['city'] ?? 'Chicago';
    $state = $article['state'] ?? 'IL';
    
    // Set coordinates based on city
    $coordinates = [
        'CHICAGO' => ['lat' => 41.8781, 'lng' => -87.6298],
        'NEW YORK' => ['lat' => 40.7128, 'lng' => -74.0060],
        'LOS ANGELES' => ['lat' => 34.0522, 'lng' => -118.2437]
    ];
    
    $cityKey = strtoupper($city);
    $coords = $coordinates[$cityKey] ?? $coordinates['CHICAGO'];
    
    return [
        'latitude' => $coords['lat'],
        'longitude' => $coords['lng'],
        'city' => $city,
        'state' => $state,
        'streetAddress' => 'Restaurant Location',
        'postalCode' => '60601'
    ];
}

function generateArticleSchema($article, $addressData) {
    $baseUrl = 'http://160.153.178.131/_sites/cleankitchens';
    $restaurantName = $article['establishment_name'] ?? 'Restaurant';
    
    // Build address object
    $address = [
        '@type' => 'PostalAddress',
        'streetAddress' => $addressData['streetAddress'],
        'addressLocality' => $addressData['city'],
        'addressRegion' => $addressData['state'],
        'postalCode' => $addressData['postalCode'],
        'addressCountry' => 'US'
    ];
    
    // Create comprehensive schema array
    $schemas = [
        // Main NewsArticle Schema
        [
            '@context' => 'https://schema.org',
            '@type' => ['NewsArticle', 'LearningResource'],
            '@id' => $baseUrl . '/' . $article['slug'],
            'headline' => $article['title'],
            'description' => $article['meta_description'] ?? '',
            'image' => [
                '@type' => 'ImageObject',
                'url' => $baseUrl . getOptimizedImageUrl($article['image_url'], 'schema'),
                'width' => 1200,
                'height' => 630
            ],
            'datePublished' => $article['published_date'] ?? date('c'),
            'dateModified' => $article['published_date'] ?? date('c'),
            'author' => [
                '@type' => 'Organization',
                'name' => 'CleanKitchens Editorial Team'
            ],
            'publisher' => [
                '@type' => 'Organization',
                'name' => 'CleanKitchens',
                'logo' => [
                    '@type' => 'ImageObject',
                    'url' => $baseUrl . '/logo.png'
                ],
                'foundingDate' => '2024',
                'sameAs' => [
                    'https://twitter.com/cleankitchens',
                    'https://facebook.com/cleankitchens'
                ]
            ],
            'mainEntityOfPage' => $baseUrl . '/' . $article['slug'],
            'about' => [
                '@type' => 'Restaurant',
                '@id' => '#restaurant-' . md5($restaurantName . $address['streetAddress']),
                'name' => $restaurantName,
                'address' => $address
            ],
            'mentions' => [
                [
                    '@type' => 'Event',
                    'name' => 'Health Inspection',
                    'startDate' => date('Y-m-d', strtotime($article['published_date'] ?? 'now')),
                    'description' => 'Restaurant health inspection conducted by Chicago Department of Public Health',
                    'location' => [
                        '@type' => 'Place',
                        'name' => $restaurantName,
                        'address' => $address
                    ]
                ]
            ],
            'educationalUse' => 'Food Safety Education',
            'teaches' => 'Restaurant Health Standards',
            'learningResourceType' => 'Case Study',
            'keywords' => 'food safety, health violation, restaurant inspection, ' . $addressData['city'],
            'articleSection' => 'Health Violations',
            'wordCount' => str_word_count(strip_tags($article['content'] ?? ''))
        ],
        
        // BreadcrumbList Schema
        [
            '@context' => 'https://schema.org',
            '@type' => 'BreadcrumbList',
            'itemListElement' => [
                [
                    '@type' => 'ListItem',
                    'position' => 1,
                    'name' => 'Home',
                    'item' => $baseUrl . '/'
                ],
                [
                    '@type' => 'ListItem',
                    'position' => 2,
                    'name' => $addressData['city'],
                    'item' => $baseUrl . '/city/' . strtolower($addressData['city'])
                ],
                [
                    '@type' => 'ListItem',
                    'position' => 3,
                    'name' => $article['title'],
                    'item' => $baseUrl . '/' . $article['slug']
                ]
            ]
        ],
        
        // Restaurant Schema - Enhanced Business Entity
        [
            '@context' => 'https://schema.org',
            '@type' => 'Restaurant',
            '@id' => '#restaurant-entity-' . md5($restaurantName . $address['streetAddress']),
            'name' => $restaurantName,
            'address' => $address,
            'servesCuisine' => 'American',
            'priceRange' => '$',
            'geo' => [
                '@type' => 'GeoCoordinates',
                'latitude' => $addressData['latitude'],
                'longitude' => $addressData['longitude']
            ],
            'areaServed' => [
                '@type' => 'City',
                'name' => $addressData['city']
            ],
            'hasOfferCatalog' => [
                '@type' => 'OfferCatalog',
                'name' => 'Food Service',
                'itemListElement' => [
                    [
                        '@type' => 'Offer',
                        'itemOffered' => [
                            '@type' => 'FoodService',
                            'name' => 'Restaurant Dining'
                        ]
                    ]
                ]
            ],
            'knowsAbout' => ['Food Safety', 'Health Inspections', 'Restaurant Operations'],
            'subjectOf' => [
                '@type' => 'NewsArticle',
                '@id' => $baseUrl . '/' . $article['slug']
            ]
        ],
        
        // CleanKitchens Organization Schema - Standalone Entity for E-A-T
        [
            '@context' => 'https://schema.org',
            '@type' => 'Organization',
            '@id' => '#cleankitchens-organization',
            'name' => 'CleanKitchens',
            'url' => $baseUrl,
            'logo' => [
                '@type' => 'ImageObject',
                'url' => $baseUrl . '/logo.png',
                'width' => 400,
                'height' => 400
            ],
            'foundingDate' => '2024',
            'description' => 'CleanKitchens provides comprehensive restaurant health inspection news and food safety education to help consumers make informed dining decisions. Our mission is to promote food safety awareness and transparency in restaurant health inspections.',
            'knowsAbout' => [
                'Restaurant Health Inspections',
                'Food Safety Regulations', 
                'Health Department Violations',
                'Restaurant Compliance',
                'Food Service Industry',
                'Public Health',
                'Consumer Protection',
                'Health Inspection Analysis',
                'Food Safety Education',
                'Restaurant Industry Reporting',
                'Government Data Interpretation'
            ],
            'areaServed' => [
                '@type' => 'Country',
                'name' => 'United States'
            ],
            'sameAs' => [
                'https://twitter.com/cleankitchens',
                'https://facebook.com/cleankitchens'
            ],
            'contactPoint' => [
                '@type' => 'ContactPoint',
                'contactType' => 'Editorial',
                'url' => $baseUrl . '/contact'
            ],
            'publishingPrinciples' => $baseUrl . '/about',
            'mainEntityOfPage' => $baseUrl,
            'hasOfferCatalog' => [
                '@type' => 'OfferCatalog',
                'name' => 'Educational Resources',
                'itemListElement' => [
                    [
                        '@type' => 'Offer',
                        'itemOffered' => [
                            '@type' => 'EducationalOccupationalCredential',
                            'name' => 'Food Safety Education'
                        ]
                    ]
                ]
            ]
        ],
        
        // FAQ Schema
        [
            '@context' => 'https://schema.org',
            '@type' => 'FAQPage',
            'mainEntity' => [
                [
                    '@type' => 'Question',
                    'name' => 'When was this restaurant last inspected?',
                    'acceptedAnswer' => [
                        '@type' => 'Answer',
                        'text' => 'This restaurant was inspected on ' . date('F j, Y', strtotime($article['published_date'] ?? 'now')) . ' by the ' . $addressData['city'] . ' health department.'
                    ]
                ],
                [
                    '@type' => 'Question',
                    'name' => 'What was the inspection result?',
                    'acceptedAnswer' => [
                        '@type' => 'Answer',
                        'text' => 'The restaurant failed its health inspection due to critical violations that posed immediate health risks to customers.'
                    ]
                ],
                [
                    '@type' => 'Question',
                    'name' => 'What does a failed inspection mean?',
                    'acceptedAnswer' => [
                        '@type' => 'Answer',
                        'text' => 'A failed inspection in Chicago\'s Pass/Fail system indicates critical food safety violations that pose immediate health risks and require immediate correction.'
                    ]
                ],
                [
                    '@type' => 'Question',
                    'name' => 'What violations were found?',
                    'acceptedAnswer' => [
                        '@type' => 'Answer',
                        'text' => 'The specific violations included issues with food temperature control, cleanliness standards, and proper food handling procedures. Details are provided in the full inspection report.'
                    ]
                ]
            ]
        ]
    ];
    
    return json_encode($schemas, JSON_UNESCAPED_SLASHES);
}