<?php
/**
 * San Francisco Inspection Data API Receiver v2
 * Receives inspection data and PDFs from Windows collector
 */

// Configuration
define('API_KEY', 'sk-sf-inspections-2025'); // Change this to a secure key
define('DATA_DIR', __DIR__ . '/');
define('PDF_DIR', DATA_DIR . 'pdfs/');
define('JSON_DIR', DATA_DIR . 'json/');
define('LOG_FILE', DATA_DIR . 'api_log.txt');

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 0);

// Set JSON response header
header('Content-Type: application/json');

// Create directories if they don't exist
if (!file_exists(DATA_DIR)) mkdir(DATA_DIR, 0755, true);
if (!file_exists(PDF_DIR)) mkdir(PDF_DIR, 0755, true);
if (!file_exists(JSON_DIR)) mkdir(JSON_DIR, 0755, true);

// Log function
function logMessage($message) {
    $timestamp = date('Y-m-d H:i:s');
    $log_entry = "[$timestamp] $message\n";
    file_put_contents(LOG_FILE, $log_entry, FILE_APPEND);
}

// Validate API key
function validateApiKey() {
    $headers = getallheaders();
    $provided_key = $headers['X-API-Key'] ?? $_SERVER['HTTP_X_API_KEY'] ?? '';
    
    if ($provided_key !== API_KEY) {
        logMessage("Invalid API key attempt: $provided_key");
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized', 'success' => false]);
        exit;
    }
    return true;
}

// Main processing
try {
    // Validate API key
    validateApiKey();
    
    // Get JSON payload
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    
    if (!$data) {
        throw new Exception('Invalid JSON payload');
    }
    
    // Check action
    $action = $data['action'] ?? '';
    
    if ($action !== 'upload_inspection') {
        throw new Exception('Invalid action');
    }
    
    // Extract data
    $inspection_id = $data['inspection_id'] ?? '';
    $facility_name = $data['facility_name'] ?? 'Unknown';
    $inspection_data = $data['inspection_data'] ?? [];
    $pdf_base64 = $data['pdf_base64'] ?? '';
    $pdf_filename = $data['pdf_filename'] ?? '';
    
    if (!$inspection_id) {
        throw new Exception('Missing inspection ID');
    }
    
    // Sanitize inspection ID for filename
    $safe_id = preg_replace('/[^A-Za-z0-9\-_]/', '', $inspection_id);
    
    // Save inspection data as JSON
    $json_file = JSON_DIR . $safe_id . '.json';
    $inspection_data['received_at'] = date('Y-m-d H:i:s');
    $inspection_data['source_ip'] = $_SERVER['REMOTE_ADDR'];
    
    if (file_put_contents($json_file, json_encode($inspection_data, JSON_PRETTY_PRINT))) {
        logMessage("Saved JSON for inspection: $inspection_id ($facility_name)");
    } else {
        throw new Exception('Failed to save JSON data');
    }
    
    // Save PDF if provided
    $pdf_saved = false;
    if ($pdf_base64 && $pdf_filename) {
        $pdf_content = base64_decode($pdf_base64);
        if ($pdf_content) {
            $pdf_file = PDF_DIR . $safe_id . '.pdf';
            if (file_put_contents($pdf_file, $pdf_content)) {
                $pdf_saved = true;
                logMessage("Saved PDF for inspection: $inspection_id (Size: " . strlen($pdf_content) . " bytes)");
            } else {
                logMessage("Failed to save PDF for inspection: $inspection_id");
            }
        }
    }
    
    // Return success response
    $response = [
        'success' => true,
        'message' => 'Inspection data received',
        'inspection_id' => $inspection_id,
        'json_saved' => true,
        'pdf_saved' => $pdf_saved,
        'timestamp' => date('Y-m-d H:i:s')
    ];
    
    echo json_encode($response);
    logMessage("Successfully processed inspection: $inspection_id");
    
} catch (Exception $e) {
    http_response_code(400);
    $error_response = [
        'success' => false,
        'error' => $e->getMessage()
    ];
    echo json_encode($error_response);
    logMessage("Error: " . $e->getMessage());
}
?>