<?php
/**
 * Database Admin API
 * Provides backend for Weaviate database management interface
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE');

// Configuration
$WEAVIATE_HOST = 'localhost';
$WEAVIATE_PORT = '8080';
$WEAVIATE_URL = "http://{$WEAVIATE_HOST}:{$WEAVIATE_PORT}";

// Get action from request (support both web and CLI)
if (php_sapi_name() === 'cli') {
    // Parse CLI arguments
    parse_str(implode('&', array_slice($argv, 1)), $_GET);
}
$action = $_GET['action'] ?? '';
$collection = $_GET['collection'] ?? '';

/**
 * Make Weaviate REST API request
 */
function weaviateRequest($endpoint, $method = 'GET', $data = null) {
    global $WEAVIATE_URL;
    
    $url = $WEAVIATE_URL . $endpoint;
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
    
    if ($data) {
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode >= 200 && $httpCode < 300) {
        return json_decode($response, true);
    } else {
        return ['error' => "HTTP $httpCode", 'response' => $response];
    }
}

/**
 * Get collection counts
 */
function getCollectionCounts() {
    $counts = [];
    
    // Get ChicagoTemp count
    $response = weaviateRequest('/v1/graphql', 'POST', [
        'query' => '{
            Aggregate {
                ChicagoTemp {
                    meta {
                        count
                    }
                }
            }
        }'
    ]);
    
    if (isset($response['data']['Aggregate']['ChicagoTemp'])) {
        $counts['ChicagoTemp'] = $response['data']['Aggregate']['ChicagoTemp'][0]['meta']['count'];
    }
    
    // Get Articles count
    $response = weaviateRequest('/v1/graphql', 'POST', [
        'query' => '{
            Aggregate {
                Articles {
                    meta {
                        count
                    }
                }
            }
        }'
    ]);
    
    if (isset($response['data']['Aggregate']['Articles'])) {
        $counts['Articles'] = $response['data']['Aggregate']['Articles'][0]['meta']['count'];
    }
    
    return $counts;
}

/**
 * Get collection records with optional filtering
 */
function getCollectionRecords($collection, $limit = 100, $offset = 0, $filter = null) {
    if (!in_array($collection, ['ChicagoTemp', 'Articles'])) {
        return ['error' => 'Invalid collection'];
    }
    
    // Build GraphQL query
    $whereClause = '';
    if ($filter) {
        // Build where clause based on filter
        if (!empty($filter['status'])) {
            $whereClause = sprintf('where: {path: ["status"], operator: Equal, valueText: "%s"}', $filter['status']);
        }
    }
    
    $query = sprintf('{
        Get {
            %s(%s limit: %d, offset: %d) {
                _additional {
                    id
                }
                inspection_id
                facility_name
                aka_name
                license_number
                facility_type
                risk_level
                address
                city
                state
                zip_code
                inspection_date
                inspection_type
                results
                violations
                latitude
                longitude
                status
                source_city
                is_combined
                combined_from_rows
                created_at
                processed_at
                published_date
                title
                slug
                excerpt
                meta_description
                tags
                cuisine_type
                service_style
                price_range
                is_chain
                chain_name
                violation_category
                violation_severity
                violation_count
                is_closure
                neighborhood
                error_log
            }
        }
    }', $collection, $whereClause, $limit, $offset);
    
    $response = weaviateRequest('/v1/graphql', 'POST', ['query' => $query]);
    
    if (isset($response['data']['Get'][$collection])) {
        $records = $response['data']['Get'][$collection];
        
        // Add ID to each record
        foreach ($records as &$record) {
            if (isset($record['_additional']['id'])) {
                $record['id'] = $record['_additional']['id'];
                unset($record['_additional']);
            }
        }
        
        // Calculate stats
        $stats = calculateStats($records);
        
        return [
            'records' => $records,
            'stats' => $stats,
            'total' => $stats['total'],
            'limit' => $limit,
            'offset' => $offset
        ];
    }
    
    return ['error' => 'Failed to fetch records', 'response' => $response];
}

/**
 * Calculate statistics from records
 */
function calculateStats($records) {
    $stats = [
        'total' => count($records),
        'pending' => 0,
        'processing' => 0,
        'processed' => 0,
        'completed' => 0,
        'error' => 0
    ];
    
    foreach ($records as $record) {
        $status = $record['status'] ?? 'unknown';
        if (isset($stats[$status])) {
            $stats[$status]++;
        }
    }
    
    return $stats;
}

/**
 * Export collection to CSV
 */
function exportToCSV($collection) {
    if (!in_array($collection, ['ChicagoTemp', 'Articles'])) {
        return ['error' => 'Invalid collection'];
    }
    
    // Get all records (up to 10000)
    $data = getCollectionRecords($collection, 10000, 0);
    
    if (isset($data['error'])) {
        return $data;
    }
    
    $records = $data['records'];
    
    if (empty($records)) {
        return ['error' => 'No records to export'];
    }
    
    // Prepare CSV
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $collection . '_export_' . date('Y-m-d_His') . '.csv"');
    
    $output = fopen('php://output', 'w');
    
    // Headers
    $headers = array_keys($records[0]);
    fputcsv($output, $headers);
    
    // Data
    foreach ($records as $record) {
        $row = [];
        foreach ($headers as $header) {
            $value = $record[$header] ?? '';
            if (is_array($value)) {
                $value = json_encode($value);
            }
            $row[] = $value;
        }
        fputcsv($output, $row);
    }
    
    fclose($output);
    exit;
}

/**
 * Get single record details
 */
function getRecord($collection, $id) {
    if (!in_array($collection, ['ChicagoTemp', 'Articles'])) {
        return ['error' => 'Invalid collection'];
    }
    
    $response = weaviateRequest("/v1/objects/{$collection}/{$id}");
    
    if (isset($response['properties'])) {
        $response['properties']['id'] = $response['id'];
        return $response['properties'];
    }
    
    return ['error' => 'Record not found'];
}

/**
 * Update record
 */
function updateRecord($collection, $id, $data) {
    if (!in_array($collection, ['ChicagoTemp', 'Articles'])) {
        return ['error' => 'Invalid collection'];
    }
    
    $response = weaviateRequest("/v1/objects/{$collection}/{$id}", 'PATCH', [
        'properties' => $data
    ]);
    
    return $response;
}

/**
 * Delete record
 */
function deleteRecord($collection, $id) {
    if (!in_array($collection, ['ChicagoTemp', 'Articles'])) {
        return ['error' => 'Invalid collection'];
    }
    
    $response = weaviateRequest("/v1/objects/{$collection}/{$id}", 'DELETE');
    
    return ['success' => true];
}

/**
 * Search records
 */
function searchRecords($collection, $query, $limit = 50) {
    if (!in_array($collection, ['ChicagoTemp', 'Articles'])) {
        return ['error' => 'Invalid collection'];
    }
    
    $graphqlQuery = sprintf('{
        Get {
            %s(
                where: {
                    operator: Or
                    operands: [
                        {path: ["facility_name"], operator: Like, valueText: "*%s*"}
                        {path: ["address"], operator: Like, valueText: "*%s*"}
                        {path: ["violations"], operator: Like, valueText: "*%s*"}
                    ]
                }
                limit: %d
            ) {
                _additional {
                    id
                }
                facility_name
                inspection_date
                status
                results
                violations
            }
        }
    }', $collection, $query, $query, $query, $limit);
    
    $response = weaviateRequest('/v1/graphql', 'POST', ['query' => $graphqlQuery]);
    
    if (isset($response['data']['Get'][$collection])) {
        $records = $response['data']['Get'][$collection];
        
        foreach ($records as &$record) {
            if (isset($record['_additional']['id'])) {
                $record['id'] = $record['_additional']['id'];
                unset($record['_additional']);
            }
        }
        
        return ['records' => $records];
    }
    
    return ['error' => 'Search failed'];
}

// Main routing
try {
    switch ($action) {
        case 'counts':
            echo json_encode(getCollectionCounts());
            break;
            
        case 'list':
            $limit = isset($_GET['limit']) ? intval($_GET['limit']) : 100;
            $offset = isset($_GET['offset']) ? intval($_GET['offset']) : 0;
            $filter = null;
            
            if (isset($_GET['status'])) {
                $filter = ['status' => $_GET['status']];
            }
            
            echo json_encode(getCollectionRecords($collection, $limit, $offset, $filter));
            break;
            
        case 'get':
            $id = $_GET['id'] ?? '';
            echo json_encode(getRecord($collection, $id));
            break;
            
        case 'update':
            $id = $_GET['id'] ?? '';
            $data = json_decode(file_get_contents('php://input'), true);
            echo json_encode(updateRecord($collection, $id, $data));
            break;
            
        case 'delete':
            $id = $_GET['id'] ?? '';
            echo json_encode(deleteRecord($collection, $id));
            break;
            
        case 'search':
            $query = $_GET['q'] ?? '';
            echo json_encode(searchRecords($collection, $query));
            break;
            
        case 'export':
            exportToCSV($collection);
            break;
            
        default:
            echo json_encode(['error' => 'Invalid action']);
    }
} catch (Exception $e) {
    echo json_encode(['error' => $e->getMessage()]);
}
?>