<?php

// Initialize Weaviate client v4
function getWeaviateClient() {
    // Using direct GraphQL API calls instead of PHP client for compatibility
    return new WeaviateGraphQL();
}

class WeaviateGraphQL {
    private $url = 'http://localhost:8080/v1/graphql';
    
    public function query($query) {
        $ch = curl_init($this->url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode(['query' => $query]));
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            throw new Exception("GraphQL query failed with HTTP $httpCode");
        }
        
        return json_decode($response, true);
    }
}

// Get dashboard statistics
function getDashboardStats() {
    $client = getWeaviateClient();
    
    // Get total articles
    $articlesQuery = '{
        Aggregate {
            Articles {
                meta {
                    count
                }
            }
        }
    }';
    
    try {
        $articlesResult = $client->query($articlesQuery);
        $totalArticles = $articlesResult['data']['Aggregate']['Articles'][0]['meta']['count'] ?? 0;
    } catch (Exception $e) {
        $totalArticles = 0;
    }
    
    // Get total tags  
    $tagsQuery = '{
        Aggregate {
            TagPages {
                meta {
                    count
                }
            }
        }
    }';
    
    try {
        $tagsResult = $client->query($tagsQuery);
        $totalTags = $tagsResult['data']['Aggregate']['TagPages'][0]['meta']['count'] ?? 0;
    } catch (Exception $e) {
        $totalTags = 0;
    }
    
    // Mock data for now (will implement real analytics later)
    return [
        'total_articles' => $totalArticles,
        'total_views' => rand(50000, 100000),
        'today_views' => rand(500, 2000),
        'total_tags' => $totalTags,
        'chart_labels' => ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
        'chart_data' => [rand(800,1200), rand(900,1300), rand(1000,1400), rand(1100,1500), rand(1200,1600), rand(600,900), rand(500,800)]
    ];
}

// Get server health metrics
function getServerHealth() {
    // CPU usage
    $load = sys_getloadavg();
    $cpuUsage = min(100, round($load[0] * 10)); // Simple approximation
    
    // Memory usage
    $memInfo = file_get_contents('/proc/meminfo');
    preg_match('/MemTotal:\s+(\d+)/', $memInfo, $totalMatch);
    preg_match('/MemAvailable:\s+(\d+)/', $memInfo, $availMatch);
    $memoryUsage = round(100 - ($availMatch[1] / $totalMatch[1] * 100));
    
    // Disk usage
    $diskTotal = disk_total_space('/');
    $diskFree = disk_free_space('/');
    $diskUsage = round(100 - ($diskFree / $diskTotal * 100));
    
    // Check Weaviate status
    $weaviateStatus = 'offline';
    try {
        $ch = curl_init('http://localhost:8080/v1/.well-known/ready');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 2);
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode === 200) {
            $weaviateStatus = 'online';
        }
    } catch (Exception $e) {
        // Keep offline status
    }
    
    $status = 'healthy';
    if ($cpuUsage > 80 || $memoryUsage > 80 || $diskUsage > 80) {
        $status = 'warning';
    }
    if ($cpuUsage > 95 || $memoryUsage > 95 || $diskUsage > 95 || $weaviateStatus === 'offline') {
        $status = 'critical';
    }
    
    return [
        'status' => $status,
        'cpu_usage' => $cpuUsage,
        'memory_usage' => $memoryUsage,
        'disk_usage' => $diskUsage,
        'weaviate_status' => $weaviateStatus
    ];
}

// Get recent articles
function getRecentArticles($limit = 5) {
    $client = getWeaviateClient();
    
    $query = sprintf('{
        Get {
            Articles(limit: %d, sort: [{path: ["published_date"], order: desc}]) {
                _additional {
                    id
                }
                title
                slug
                published_date
                city
                state
            }
        }
    }', $limit);
    
    $articles = [];
    
    try {
        $result = $client->query($query);
        
        if (isset($result['data']['Get']['Articles'])) {
            foreach ($result['data']['Get']['Articles'] as $article) {
                $articles[] = [
                    'id' => $article['_additional']['id'] ?? '',
                    'title' => $article['title'] ?? 'Untitled',
                    'slug' => $article['slug'] ?? '',
                    'published_date' => isset($article['published_date']) ? date('M j, Y', strtotime($article['published_date'])) : 'Unknown',
                    'views' => rand(100, 5000) // Mock data for now
                ];
            }
        }
    } catch (Exception $e) {
        error_log("Failed to get recent articles: " . $e->getMessage());
    }
    
    return $articles;
}

// Get popular articles
function getPopularArticles($limit = 5) {
    // Mock data for now - will implement real analytics later
    $articles = getRecentArticles($limit);
    foreach ($articles as &$article) {
        $article['views'] = rand(5000, 20000);
        $article['avg_time'] = rand(30, 180);
    }
    usort($articles, function($a, $b) {
        return $b['views'] - $a['views'];
    });
    return $articles;
}

// Get pipeline status
function getPipelineStatus() {
    $lastLogFile = '/var/www/twin-digital-media/public_html/_sites/cleankitchens/production/scripts/logs/processor.log';
    $lastFetch = 'Never';
    
    if (file_exists($lastLogFile)) {
        $lastFetch = date('M j, Y g:i A', filemtime($lastLogFile));
    }
    
    // Check if processor is running
    $status = 'idle';
    exec("ps aux | grep 'comprehensive_processor.py' | grep -v grep", $output);
    if (!empty($output)) {
        $status = 'running';
    }
    
    return [
        'last_fetch' => $lastFetch,
        'processed_today' => rand(50, 200),
        'articles_today' => rand(10, 50),
        'next_run' => date('M j, Y', strtotime('+1 day')) . ' 2:00 AM',
        'status' => $status
    ];
}

// Get article by ID
function getArticleById($id) {
    $client = getWeaviateClient();
    
    $result = $client->graphql()->get()
        ->withClassName('Articles')
        ->withFields('id title slug content excerpt meta_description meta_keywords 
                     published_date last_updated establishment_name city state 
                     zip_code inspection_date inspection_type results violations 
                     auto_tags image_url latitude longitude')
        ->withWhere([
            'path' => ['id'],
            'operator' => 'Equal',
            'valueString' => $id
        ])
        ->withLimit(1)
        ->run();
    
    if (isset($result['data']['Get']['Articles'][0])) {
        return $result['data']['Get']['Articles'][0];
    }
    
    return null;
}

// Update article
function updateArticle($id, $data) {
    $client = getWeaviateClient();
    
    try {
        $client->dataObject()->update($data, 'Articles', $id);
        return true;
    } catch (Exception $e) {
        error_log("Failed to update article: " . $e->getMessage());
        return false;
    }
}

// Delete article
function deleteArticle($id) {
    $client = getWeaviateClient();
    
    try {
        $client->dataObject()->delete($id, 'Articles');
        return true;
    } catch (Exception $e) {
        error_log("Failed to delete article: " . $e->getMessage());
        return false;
    }
}

// Create new article
function createArticle($data) {
    $client = getWeaviateClient();
    
    // Generate slug if not provided
    if (empty($data['slug'])) {
        $data['slug'] = generateSlug($data['title']);
    }
    
    // Set timestamps
    $data['published_date'] = $data['published_date'] ?? date('c');
    $data['last_updated'] = date('c');
    
    try {
        $id = $client->dataObject()->create($data, 'Articles');
        return $id;
    } catch (Exception $e) {
        error_log("Failed to create article: " . $e->getMessage());
        return false;
    }
}

// Generate slug from title
function generateSlug($title) {
    $slug = strtolower(trim($title));
    $slug = preg_replace('/[^a-z0-9-]/', '-', $slug);
    $slug = preg_replace('/-+/', '-', $slug);
    $slug = trim($slug, '-');
    
    // Add date prefix
    $datePrefix = date('mdy');
    return $datePrefix . '-' . substr($slug, 0, 80);
}

// Get all tags
function getAllTags() {
    $client = getWeaviateClient();
    
    $result = $client->graphql()->get()
        ->withClassName('TagPages')
        ->withFields('id tag_name slug article_count')
        ->withSort(['path' => ['article_count'], 'order' => 'desc'])
        ->withLimit(100)
        ->run();
    
    $tags = [];
    if (isset($result['data']['Get']['TagPages'])) {
        foreach ($result['data']['Get']['TagPages'] as $tag) {
            $tags[] = $tag;
        }
    }
    
    return $tags;
}

// Get analytics data
function getAnalyticsData($days = 7) {
    // This will be implemented with real analytics tracking
    // For now, return mock data
    $data = [];
    for ($i = $days; $i >= 0; $i--) {
        $date = date('Y-m-d', strtotime("-$i days"));
        $data['dates'][] = date('M j', strtotime($date));
        $data['pageviews'][] = rand(800, 2000);
        $data['sessions'][] = rand(400, 1000);
        $data['users'][] = rand(300, 800);
    }
    
    return $data;
}
?>