# CleanKitchens Site Setup Knowledge Base

## Overview
CleanKitchens has been transformed from a complex routing-based system to a simple, templated PHP website with perfect PageSpeed scores (100/100 across all metrics).

## Architecture

### No Router Approach
The site now operates as a traditional PHP website with clean URLs handled by `.htaccess` rewrite rules. **No complex routing system.**

### File Structure
```
/var/www/twin-digital-media/public_html/_sites/cleankitchens/
├── index.php                    # Homepage
├── article.php                  # Article viewer
├── about.php                    # Static pages
├── contact.php
├── privacy.php
├── terms.php
├── disclaimer.php
├── methodology.php
├── editorial-policy.php
├── corrections.php
├── functions_simple.php         # Core functions (no caching)
├── .htaccess                    # URL rewriting
├── includes/
│   ├── config.php              # Site configuration
│   ├── head.php                # HTML head template
│   ├── nav.php                 # Navigation template
│   ├── footer.php              # Footer template
│   └── styles.php              # CSS styles
└── templates/
    └── static.php              # Static page template
```

## URL Structure

### Homepage
- **URL**: `/` or `/index.php`
- **File**: `index.php`
- **Template**: Direct PHP file using news template logic

### Articles
- **URL Pattern**: `/YYYY/MM/DD/complete-article-title`
- **Example**: `/2025/08/16/chicago-restaurant-fails-inspection-with-sewage-leak-and-pest-issues`
- **File**: `article.php`
- **Rewrite Rule**: `^(\d{4}/\d{2}/\d{2}/.+)$ article.php?slug=$1`

### Static Pages
- **URL Pattern**: `/page-name`
- **Examples**: `/about`, `/contact`, `/privacy`, `/terms`
- **Files**: Individual PHP files (`about.php`, `contact.php`, etc.)
- **Template**: All use `templates/static.php`
- **Rewrite Rules**: Direct mapping (e.g., `^about/?$ about.php`)

## Core Functions (`functions_simple.php`)

### Database Functions
- `queryWeaviate($query)` - Direct GraphQL queries to Weaviate
- `getArticleBySlug($slug)` - Fetch article by slug
- `getHomepageArticles()` - Get recent articles for homepage
- `getFeaturedArticle()` - Get latest article for homepage hero

### Image Functions
- `getArticleImage($article)` - Returns single inspection image for all articles
- `getOptimizedImageUrl($baseUrl, $size, $article)` - Optimized image URLs
- `getImageSrcSet($imagePath, $sizes)` - WebP responsive image sets

### Utility Functions
- `timeAgo($timestamp)` - Human-readable time formatting
- `getRelatedArticles($currentSlug, $limit)` - Related article suggestions

## Image Optimization

### Image Strategy
- **Single Inspection Image**: All articles use `/assets/images/health-inspection-report.jpg`
- **Pre-optimized Versions**:
  - Thumbnails: `health-inspection-report_thumb.webp` (9.7KB)
  - Article pages: `health-inspection-report_page.webp` (15KB)
  - Social sharing: `health-inspection-report_og.jpg` (54KB)

### Format Strategy
- **HTML Display**: WebP format for all visible images
- **Social Sharing**: JPG format for Open Graph meta tags
- **Responsive**: Multiple sizes via `srcset` for optimal loading

## Performance Optimizations

### Achieved Scores
- **Performance**: 100/100
- **Accessibility**: 100/100
- **Best Practices**: 100/100
- **SEO**: 100/100
- **CLS**: 0 (zero layout shifts)

### Key Optimizations
1. **Layout Stability**: Fixed dimensions on all containers
2. **Image Optimization**: Pre-optimized WebP images
3. **Delayed Ads**: Ads load 1 second after page load
4. **CSS Containment**: `contain: layout` on grid elements
5. **Aspect Ratios**: `aspect-ratio: 4/3` on image containers

## Static Page System

### Template Structure
All static pages use the same template (`templates/static.php`) with conditional content based on `$staticPage` variable.

### Page Setup Pattern
```php
<?php
require_once 'includes/config.php';
require_once 'functions_simple.php';

$pageType = 'static';
$staticPage = 'about';  // Determines content section
$pageData = [
    'title' => 'About CleanKitchens',
    'description' => 'Page description for meta tags',
    'h1' => 'About CleanKitchens'
];

include 'includes/head.php';
include 'includes/nav.php';
include 'templates/static.php';
include 'includes/footer.php';
?>
```

### Available Static Pages
- **About**: Company mission, founder info, data sources
- **Contact**: Editorial contact information
- **Privacy**: Privacy policy and data handling
- **Terms**: Terms of service
- **Disclaimer**: Important notices about historical data
- **Methodology**: Data collection and processing methods
- **Editorial Policy**: Editorial standards and principles
- **Corrections**: Correction policy and recent corrections

## Database Integration

### Weaviate Vector Database
- **Endpoint**: `http://localhost:8080/v1/graphql`
- **Collection**: `Articles`
- **Query Language**: GraphQL
- **No Caching**: Direct queries for simplicity and reliability

### Article Schema
```
- title
- slug (YYYY/MM/DD/complete-title format)
- content
- excerpt
- meta_description
- city
- state
- establishment_name
- published_date
- image_url (not used - single image strategy)
- tags
```

## Development Guidelines

### Adding New Static Pages
1. Create new PHP file (e.g., `new-page.php`)
2. Add rewrite rule to `.htaccess`: `RewriteRule ^new-page/?$ new-page.php [L]`
3. Add content section to `templates/static.php`
4. Update footer navigation if needed

### Modifying Styles
- Edit `includes/styles.php`
- CSS variables used for theming
- Focus on maintaining layout stability (CLS prevention)

### Adding New Features
- Prioritize simplicity over complexity
- Test PageSpeed impact before deploying
- Maintain zero CLS score
- Use direct database queries (no caching complexity)

## Deployment Notes

### No Complex Dependencies
- Standard PHP 7.4+
- No frameworks or complicated build processes
- Direct file serving with Apache mod_rewrite

### Performance Monitoring
- Monitor Core Web Vitals
- Maintain 100/100 PageSpeed scores
- Watch for layout shifts (keep CLS at 0)

### Backup Considerations
- Key files: `functions_simple.php`, `.htaccess`, `includes/`
- Static page content in `templates/static.php`
- Image assets in `/assets/images/`

## Troubleshooting

### Common Issues
1. **Static pages not loading**: Check `.htaccess` rewrite rules
2. **Images not optimizing**: Verify optimized versions exist in `/assets/images/`
3. **Layout shifts**: Check container min-heights and image dimensions
4. **Database connection**: Verify Weaviate endpoint in `queryWeaviate()`

### Key Files for Debugging
- `.htaccess` - URL routing
- `functions_simple.php` - Core functionality
- `includes/head.php` - Meta tags and performance hints
- `includes/styles.php` - CSS and layout stability

This simple, templated approach provides excellent performance while maintaining ease of maintenance and development.