<?php

/**

 * Search Page Logic - SoberList.org

 * 

 * This file contains all the PHP logic needed for the search page functionality.

 * It processes search parameters and fetches the results from the database.

 */



// Debug function - only outputs in development mode

function debug_output($message, $type = 'info') {

    // For critical issues, add to a log file instead of displaying

    if ($type === 'critical') {

        error_log("[SEARCH] " . $message);

    }

    return;

}



// Make sure we have a database connection

global $db;



// Parse and sanitize query parameters

$searchQuery = isset($_GET['q']) ? trim(htmlspecialchars($_GET['q'])) : '';

$location = isset($_GET['location']) ? trim(htmlspecialchars($_GET['location'])) : '';

$facility_types = isset($_GET['facility_types']) && is_array($_GET['facility_types']) 

    ? array_map('trim', array_filter($_GET['facility_types']))

    : [];

$state_code = isset($_GET['state']) ? trim(htmlspecialchars($_GET['state'])) : '';

$gender = isset($_GET['gender']) ? trim(htmlspecialchars($_GET['gender'])) : '';

$payment_option = isset($_GET['payment']) ? trim(htmlspecialchars($_GET['payment'])) : '';

$radius = isset($_GET['radius']) ? filter_var($_GET['radius'], FILTER_VALIDATE_INT, 

          ['options' => ['default' => 10, 'min_range' => 5, 'max_range' => 100]]) : 10;

$page = isset($_GET['pg']) ? max(1, filter_var($_GET['pg'], FILTER_VALIDATE_INT, 

       ['options' => ['default' => 1, 'min_range' => 1]])) : 1;

$sort = isset($_GET['sort']) ? trim(htmlspecialchars($_GET['sort'])) : 'relevance';

$view = isset($_GET['view']) && in_array($_GET['view'], ['list', 'map']) ? $_GET['view'] : 'list';



// Ensure valid radius and force to integer

$radius = (int)$radius;

if (!$radius || $radius < 5) $radius = 10;

if ($radius > 100) $radius = 100;



// Build search parameters array (useful for pagination links)

$search_params = [

    'q' => $searchQuery,

    'location' => $location,

    'view' => $view

];



if (!empty($facility_types)) {

    $search_params['facility_types'] = $facility_types;

}

if (!empty($state_code)) {

    $search_params['state'] = $state_code;

}

if (!empty($gender)) {

    $search_params['gender'] = $gender;

}

if (!empty($payment_option)) {

    $search_params['payment'] = $payment_option;

}

if ($radius != 10) {

    $search_params['radius'] = $radius;

}

if ($sort != 'relevance') {

    $search_params['sort'] = $sort;

}

if ($page > 1) {

    $search_params['pg'] = $page;

}



// Set pagination info

$items_per_page = 12; // Number of items per page

$offset = ($page - 1) * $items_per_page;



// Set dynamic title based on search parameters

$dynamic_title = 'Search Recovery Resources';

if (!empty($searchQuery)) {

    $dynamic_title = 'Search Results for "' . htmlspecialchars($searchQuery) . '"';

} elseif (!empty($facility_types) && count($facility_types) === 1) {

    $type_names = [

        'HH' => 'Sober Living Homes',

        'SA' => 'Treatment Centers',

        'DT' => 'Detox Facilities',

        'MH' => 'Mental Health Services',

        'SUMH' => 'Co-occurring Disorder Treatment',

        'OP' => 'Outpatient Programs'

    ];

    

    $type_name = $type_names[$facility_types[0]] ?? 'Recovery Resources';

    $dynamic_title = $type_name;

    

    if (!empty($state_code)) {

        // Get state name from code

        try {

            $stmt = $db->prepare("SELECT state_name FROM states WHERE state_code = ?");

            $stmt->execute([$state_code]);

            $state_name = $stmt->fetchColumn();

            

            if ($state_name) {

                $dynamic_title .= " in $state_name";

            }

        } catch (PDOException $e) {

            // Silent error handling

        }

    } elseif (!empty($location)) {

        $dynamic_title .= " near $location";

    }

} elseif (!empty($location)) {

    $dynamic_title = "Recovery Resources near $location";

}



// Initialize variables

$facilities = [];

$total_count = 0;

$total_pages = 1;

$map_facilities = [];

$states = [];

$payment_options = [];

$search_error = false;



// Get states for state filter dropdown

try {

    $stmt = $db->prepare("

        SELECT s.state_code, s.state_name, COUNT(f.id) as facility_count

        FROM states s

        LEFT JOIN facilities f ON s.id = f.state_id AND f.is_active = 1

        GROUP BY s.state_code, s.state_name

        ORDER BY s.state_name

    ");

    $stmt->execute();

    $states = $stmt->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {

    // Silent error handling

}



// Get payment options for dropdown

try {

    $stmt = $db->prepare("SELECT * FROM payment_options ORDER BY name ASC");

    $stmt->execute();

    $payment_options = $stmt->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {

    // Silent error handling

}



/**

 * Calculate average rating from Google reviews JSON

 */

function calculateGoogleRating($facility) {

    if (empty($facility['google_reviews'])) {

        return [

            'avg_rating' => 0,

            'total_reviews' => 0

        ];

    }

    

    $google_reviews = json_decode($facility['google_reviews'], true) ?: [];

    

    if (empty($google_reviews)) {

        return [

            'avg_rating' => 0,

            'total_reviews' => 0

        ];

    }

    

    $sum = 0;

    foreach ($google_reviews as $review) {

        $sum += $review['rating'] ?? 0;

    }

    

    return [

        'avg_rating' => count($google_reviews) > 0 ? $sum / count($google_reviews) : 0,

        'total_reviews' => count($google_reviews)

    ];

}



// Execute search if we have any query parameters and DB is connected

if (!isset($db_error) && (!empty($searchQuery) || !empty($location) || !empty($facility_types) || !empty($state_code) || !empty($gender) || !empty($payment_option))) {

    try {

        // Location search conditions

        $location_join = '';

        $location_where = '';

        $location_params = [];

        

        if (!empty($location)) {

            // Try to get coordinates for location-based search

            try {

                // First check the location_cache table

                $stmt = $db->prepare("

                    SELECT latitude, longitude 

                    FROM location_cache 

                    WHERE location_string = ? OR zip = ?

                ");

                $stmt->execute([$location, $location]);

                $location_data = $stmt->fetch(PDO::FETCH_ASSOC);

                

                // If not found in cache, try to parse as ZIP or city,state

                if (!$location_data || empty($location_data['latitude']) || empty($location_data['longitude'])) {

                    // Check if it's a ZIP code (either 5 digit or ZIP+4 format)

                    if (preg_match('/\b(\d{5}(?:-\d{4})?)\b/', $location, $matches) || 

                        (is_numeric($location) && strlen($location) >= 5)) {

                        // Extract just the 5-digit part if it's a ZIP+4

                        $zip_code = preg_match('/\b(\d{5})/', $location, $zip_matches) ? 

                                    $zip_matches[1] : substr($location, 0, 5);

                        

                        // Try looking up in location_cache by zip

                        $stmt = $db->prepare("

                            SELECT latitude, longitude 

                            FROM location_cache 

                            WHERE zip = ?

                        ");

                        $stmt->execute([$zip_code]);

                        $location_data = $stmt->fetch(PDO::FETCH_ASSOC);

                    } elseif (preg_match('/([^,]+),\s*([A-Za-z]{2})/', $location, $matches)) {

                        // Try city, state format

                        $city = trim($matches[1]);

                        $state = strtoupper(trim($matches[2]));

                        

                        // Try location_cache first

                        $stmt = $db->prepare("

                            SELECT latitude, longitude

                            FROM location_cache

                            WHERE LOWER(city) = LOWER(?) AND UPPER(state) = UPPER(?)

                        ");

                        $stmt->execute([$city, $state]);

                        $location_data = $stmt->fetch(PDO::FETCH_ASSOC);

                    }

                }

                

                if ($location_data && !empty($location_data['latitude']) && !empty($location_data['longitude'])) {

                    $lat = $location_data['latitude'];

                    $lng = $location_data['longitude'];

                    

                    $location_join = "

                        CROSS JOIN (SELECT :lat AS search_lat, :lng AS search_lng) AS loc

                    ";

                    $location_where = "

                        AND (3959 * acos(cos(radians(loc.search_lat)) * cos(radians(f.latitude)) * 

                        cos(radians(f.longitude) - radians(loc.search_lng)) + 

                        sin(radians(loc.search_lat)) * sin(radians(f.latitude)))) <= :radius

                    ";

                    $location_params = [

                        ':lat' => $lat,

                        ':lng' => $lng,

                        ':radius' => $radius

                    ];

                }

            } catch (PDOException $e) {

                // Silent error handling

                error_log("Location search error: " . $e->getMessage());

            }

        }

        

        // Build search conditions

        $conditions = ["f.is_active = 1"];

        $params = [];

        

        // Keyword search

        if (!empty($searchQuery)) {

            // Prepare search parameter for keyword search

            $search_param = '%' . strtolower($searchQuery) . '%';

            

            // Add service search for keyword

            $service_condition = "

                EXISTS (

                    SELECT 1 FROM facility_services fs

                    JOIN service_codes sc ON fs.service_code = sc.service_code

                    WHERE fs.facility_id = f.id AND (

                        LOWER(sc.service_name) LIKE :search_term_service OR

                        LOWER(sc.service_description) LIKE :search_term_service_desc

                    )

                )

            ";

            $params[':search_term_service'] = $search_param;

            $params[':search_term_service_desc'] = $search_param;

            

            $conditions[] = "(

                LOWER(f.name) LIKE :search_term_name OR 

                LOWER(f.city) LIKE :search_term_city OR 

                LOWER(f.address_line1) LIKE :search_term_addr OR 

                LOWER(f.description) LIKE :search_term_desc OR

                {$service_condition}

            )";

            

            $params[':search_term_name'] = $search_param;

            $params[':search_term_city'] = $search_param;

            $params[':search_term_addr'] = $search_param;

            $params[':search_term_desc'] = $search_param;

        }

        

        // Facility type filter

        if (!empty($facility_types)) {

            $type_placeholders = [];

            foreach ($facility_types as $index => $type) {

                $placeholder = ":type{$index}";

                $type_placeholders[] = "ct.code = {$placeholder}";

                $params[$placeholder] = $type;

            }

            

            $conditions[] = "(

                EXISTS (

                    SELECT 1 FROM facility_care_types fct

                    JOIN care_types ct ON fct.care_type_id = ct.id

                    WHERE fct.facility_id = f.id 

                    AND (" . implode(' OR ', $type_placeholders) . ")

                )

            )";

        }

        

        // State filter

        if (!empty($state_code)) {

            $conditions[] = "UPPER(s.state_code) = UPPER(:state_code)";

            $params[':state_code'] = $state_code;

            

            // If location is empty but state is provided, also search facilities by state code directly

            if (empty($location)) {

                // This helps when state dropdown is used without a specific location

                $conditions[count($conditions)-1] = "(UPPER(s.state_code) = UPPER(:state_code) OR f.state_id IN (SELECT id FROM states WHERE UPPER(state_code) = UPPER(:state_code)))";

            }

        }

        

        // Gender filter

        if (!empty($gender)) {

            $conditions[] = "f.gender_accepted = :gender";

            $params[':gender'] = $gender;

        }

        

        // Payment option filter

        if (!empty($payment_option)) {

            $conditions[] = "

                EXISTS (

                    SELECT 1 FROM facility_payment_options fpo

                    JOIN payment_options po ON fpo.payment_option_id = po.id

                    WHERE fpo.facility_id = f.id AND po.code = :payment_option

                )

            ";

            $params[':payment_option'] = $payment_option;

        }

        

        // Build query for counting total results

        $count_sql = "

            SELECT COUNT(DISTINCT f.id)

            FROM facilities f

            LEFT JOIN states s ON f.state_id = s.id

            LEFT JOIN facility_care_types fct ON f.id = fct.facility_id

            LEFT JOIN care_types ct ON fct.care_type_id = ct.id

            {$location_join}

            WHERE " . implode(' AND ', $conditions) . $location_where;

        

        // Execute count query

        $stmt = $db->prepare($count_sql);

        

        // Bind parameters for count query

        foreach (array_merge($params, $location_params) as $key => $value) {

            $stmt->bindValue($key, $value);

        }

        

        $stmt->execute();

        $total_count = $stmt->fetchColumn();

        $total_pages = ceil($total_count / $items_per_page);

        

        error_log("[SEARCH] Found {$total_count} total results, pages: {$total_pages}");

        

        // Continue only if we found results

        if ($total_count > 0) {

            // Distance calculation for search results

            $distance_select = '';

            if (!empty($location_params)) {

                $distance_select = ", 

                    (3959 * acos(cos(radians(loc.search_lat)) * cos(radians(f.latitude)) * 

                    cos(radians(f.longitude) - radians(loc.search_lng)) + 

                    sin(radians(loc.search_lat)) * sin(radians(f.latitude)))) AS distance

                ";

            }

            

            // Build sort clause

            $sort_clause = "";

            switch ($sort) {

                case 'name_asc':

                    $sort_clause = "ORDER BY f.name ASC";

                    break;

                case 'name_desc':

                    $sort_clause = "ORDER BY f.name DESC";

                    break;

                case 'rating':

                    $sort_clause = "ORDER BY COALESCE(f.avg_rating, 0) DESC, f.total_reviews DESC";

                    break;

                case 'reviews':

                    $sort_clause = "ORDER BY COALESCE(f.total_reviews, 0) DESC, f.avg_rating DESC";

                    break;

                case 'distance':

                    if (!empty($location_params)) {

                        $sort_clause = "ORDER BY distance ASC";

                    } else {

                        $sort_clause = "ORDER BY f.avg_rating DESC";

                    }

                    break;

                case 'relevance':

                default:

                    if (!empty($searchQuery)) {

                        // Relevance ordering with unique parameter names

                        $sort_clause = "

                            ORDER BY 

                                CASE 

                                    WHEN LOWER(f.name) = :exact_match THEN 100

                                    WHEN LOWER(f.name) LIKE :starts_with THEN 90

                                    WHEN LOWER(f.city) = :exact_city THEN 80

                                    ELSE 50

                                END DESC,

                                f.avg_rating DESC

                        ";

                        $params[':exact_match'] = strtolower($searchQuery);

                        $params[':starts_with'] = strtolower($searchQuery) . '%';

                        $params[':exact_city'] = strtolower($searchQuery);

                    } elseif (!empty($location_params)) {

                        $sort_clause = "ORDER BY distance ASC";

                    } else {

                        $sort_clause = "ORDER BY f.featured DESC, f.avg_rating DESC";

                    }

                    break;

            }

            

            // Group by might be causing issues - make sure your MySQL mode supports it

            $sql = "

                SELECT 

                    f.*,

                    s.state_code,

                    (SELECT file_name FROM facility_photos WHERE facility_id = f.id AND is_primary = 1 LIMIT 1) as primary_photo,

                    GROUP_CONCAT(DISTINCT ct.code SEPARATOR ',') as facility_types,

                    f.google_reviews

                    {$distance_select}

                FROM facilities f

                LEFT JOIN states s ON f.state_id = s.id

                LEFT JOIN facility_care_types fct ON f.id = fct.facility_id

                LEFT JOIN care_types ct ON fct.care_type_id = ct.id

                {$location_join}

                WHERE " . implode(' AND ', $conditions) . $location_where . "

                GROUP BY f.id

                {$sort_clause}

                LIMIT :offset, :limit

            ";

            

            error_log("[SEARCH] SQL: " . $sql);

            

            // Execute main query

            $stmt = $db->prepare($sql);

            

            // Bind pagination parameters

            $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);

            $stmt->bindValue(':limit', $items_per_page, PDO::PARAM_INT);

            

            // Bind other parameters

            foreach (array_merge($params, $location_params) as $key => $value) {

                $stmt->bindValue($key, $value);

            }

            

            $stmt->execute();

            $facilities = $stmt->fetchAll(PDO::FETCH_ASSOC);

            

            // Log critical issues with results

            error_log("[SEARCH] Query returned " . count($facilities) . " results for page {$page}");

            

            // After search is complete, ensure results are properly formatted for the template

            if (!empty($facilities)) {

                foreach ($facilities as $key => $facility) {

                    // Calculate Google rating based on reviews JSON

                    $rating_data = calculateGoogleRating($facility);

                    $facilities[$key]['avg_rating'] = $rating_data['avg_rating'];

                    $facilities[$key]['total_reviews'] = $rating_data['total_reviews'];

                    

                    // Ensure these fields exist as they're often accessed directly in the template

                    if (!isset($facility['facility_types'])) {

                        $facilities[$key]['facility_types'] = '';

                    }

                    if (!isset($facility['state_code'])) {

                        $facilities[$key]['state_code'] = '';

                    }

                    if (!isset($facility['primary_photo'])) {

                        $facilities[$key]['primary_photo'] = '';

                    }

                }

                // Log the first result for debugging

                if (count($facilities) > 0) {

                    error_log("[SEARCH] First result ID: " . $facilities[0]['id'] . ", Name: " . $facilities[0]['name']);

                }

            }

            

            // Make sure each facility has required fields

            foreach ($facilities as $key => $facility) {

                // Ensure state_code exists (required by template)

                if (empty($facility['state_code']) && !empty($facility['state_id'])) {

                    // Try to get state code from state_id

                    try {

                        $stmt = $db->prepare("SELECT state_code FROM states WHERE id = ?");

                        $stmt->execute([$facility['state_id']]);

                        $state_code = $stmt->fetchColumn();

                        if ($state_code) {

                            $facilities[$key]['state_code'] = $state_code;

                        }

                    } catch (PDOException $e) {

                        // Silent error handling

                    }

                }

            }

            

            // If map view, get additional facilities for map (limited to 200 to prevent performance issues)

            if ($view === 'map') {

                // Fix map query to match main query structure

                $map_sql = str_replace("LIMIT :offset, :limit", "LIMIT 0, 200", $sql);

                $stmt = $db->prepare($map_sql);

                

                // Bind map query parameters (except pagination)

                foreach (array_merge($params, $location_params) as $key => $value) {

                    $stmt->bindValue($key, $value);

                }

                

                $stmt->execute();

                $map_facilities = $stmt->fetchAll(PDO::FETCH_ASSOC);

                

                // Calculate Google ratings for map facilities

                if (!empty($map_facilities)) {

                    foreach ($map_facilities as $key => $facility) {

                        $rating_data = calculateGoogleRating($facility);

                        $map_facilities[$key]['avg_rating'] = $rating_data['avg_rating'];

                        $map_facilities[$key]['total_reviews'] = $rating_data['total_reviews'];

                    }

                }

                

                error_log("[SEARCH] Retrieved " . count($map_facilities) . " facilities for map view");

            } else {

                $map_facilities = $facilities;

            }

        }

        

    } catch (PDOException $e) {

        // Silent error handling in production

        $search_error = true;

        $facilities = [];

        $total_count = 0;

        $total_pages = 1;

        error_log("[SEARCH ERROR] " . $e->getMessage());

    }

}



/**

 * Helper function to get facility image URL based on type

 */

function getFacilityImage($facility) {

    $image_url = '';

    if (!empty($facility['primary_photo'])) {

        $image_url = '/uploads/facilities/' . htmlspecialchars($facility['primary_photo']);

    } else {

        // Default image based on facility type

        $facility_types = !empty($facility['facility_types']) ? explode(',', $facility['facility_types']) : [];

        $primary_type = !empty($facility_types) ? trim($facility_types[0]) : '';

        

        // Only have 4 images - 3 type-specific and 1 default

        $type_images = [

            'HH' => '/assets/images/sober-house-near-me.webp',

            'SA' => '/assets/images/treatment-near-me.webp',

            'DT' => '/assets/images/detox-near-me.webp'

        ];

        

        $image_url = isset($type_images[$primary_type]) ? $type_images[$primary_type] : '/assets/images/we-do-recover.webp';

    }

    

    return $image_url;

}



/**

 * Helper function to get badge class based on facility type

 */

function getBadgeClass($type) {

    $type = trim($type);

    $classes = [

        'HH' => 'bg-primary',

        'SA' => 'bg-success',

        'DT' => 'bg-info',

        'MH' => 'bg-warning',

        'SUMH' => 'bg-secondary',

        'OP' => 'bg-secondary'

    ];

    

    return $classes[$type] ?? 'bg-secondary';

}



/**

 * Helper function to get facility type label

 */

function getFacilityTypeLabel($type) {

    $type = trim($type);

    $labels = [

        'HH' => 'Sober Living',

        'SA' => 'Treatment',

        'DT' => 'Detox',

        'MH' => 'Mental Health',

        'SUMH' => 'Co-occurring',

        'OP' => 'Outpatient',

        'SHELTER' => 'Shelter'

    ];

    

    return $labels[$type] ?? $type;

}



/**

 * Helper function to get facility slug

 */

function getFacilitySlug($facility) {

    if (!empty($facility['slug'])) {

        return $facility['slug'];

    }

    

    // Generate a simple slug from name and location

    $name = strtolower(preg_replace('/[^a-zA-Z0-9]+/', '-', $facility['name']));

    $city = strtolower(preg_replace('/[^a-zA-Z0-9]+/', '-', $facility['city']));

    $state = strtolower($facility['state_code']);

    

    return trim($name, '-') . '-' . trim($city, '-') . '-' . $state;

}



/**

 * Include a JavaScript function to properly enable the radius slider

 */

function getRadiusSliderScript() {

    return "

    document.addEventListener('DOMContentLoaded', function() {

        // Radius Slider

        const radiusSlider = document.getElementById('radius-slider');

        const radiusValue = document.getElementById('radius-value');

        

        if (radiusSlider && radiusValue) {

            // Set initial value

            radiusValue.textContent = radiusSlider.value + 'mi';

            

            // Update radius value display when slider changes

            radiusSlider.addEventListener('input', function() {

                radiusValue.textContent = this.value + 'mi';

            });

        }

    });

    ";

}